/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at sfinney@sfinney.com.
 */

/*********************************************************************
 * Code for "triggers": user-set triggers (based on metronome count, keystroke
 * count, or elapsed time) which cause integer parameter changes, which 
 * will cause an on the fly change in behavior. These allow for turning
 * metronome on/off (for, e.g., synchronization/continuation paradigms),
 * terminating an experiment and changing or perturbing feedback.
 * The user can define up to MAXEVENTS such triggers, each of which
 * causes ONE parameter change; multiple defined triggers can be set
 * to the same event to implement multiple parameter changes.
 *
 * TRIGGER TYPES:
 * (1) Metronome triggers, based on a logical underlying beat.
 * (2) Time triggers (based on ms since trial start): these are placed in 
 * the scheduling queue at startup, and are acted upon in ProcOutput.
 * (3) Keystroke triggers: action takes place near the key-reading code; these
 * will affect feedback to the designated keystroke itself. 
 *
 *
 * A parameter file line identifying a trigger event has the following format;
 * all fields must be supplied. FLAG should always be 0, for now.
 *   EVENT <ID> <TYPE> <TRIG_CNT> <PARAM_NAME> <PARAM_VALUE>
 *
 *	TYPE is K(eyboard), M(etronome), or T(ime).
 *	ID is just a user ID number, for identifying events in the output
 * 	file, output lines, and also so that (by reusing an ID) you can 
 *	overwrite an existing trigger. 
 *	TRIG_CNT is # of down events, or millseconds for T.
 *	PARAM_NAME is the capital letter name (only) of an integer (only!) 
 *	parameter, and PARAM_VALUE is the new value  (e.g. "FDELAY 100")
 *	For now, only such integer parameters can be used. "END_EXP" with
 *	any value is a special case that will end an experiment.
 *
 * 
 * IMPLEMENTATION:
 * The global trig_event_table contains the master information for triggers.
 * Since metronome and keystroke events must be checked on the fly, we
 * maintain sorted lists (separately) for those event types which point
 * to the master trig_event_table; then you just need to check the first entry
 * in those tables.
 * 
 * OUTPUT FORMAT: Event triggers  go into the output file, and reuse some
 * fields. The "type" will be "E". The "updown" field will identify the
 * event subtype ("K", "T", "M"). The "vel" field will contain the (user-
 * supplied) identifier (which allows correlation of the event with the
 * parameter printout). Key events will also have the sequence number of
 * the keystroke. The note number will be the (internal) index, which will
 * not be of much use, but it will be non-zero. Channel number will be 0.
 *
 * NOTES:
 * (1) Only integer parameters can be changed, not character, array, or
 * trigger events themselves. Not  all integer parameters
 * have an effect; the masking parameters and stdout are known exceptions.
 * (2) If you are running in interactive mode, a parameter value changed by
 * a trigger will STAY CHANGED. 
 * (3) Metronome or time based trigger events may possibly occur between
 * a key down and a key up, potentially leading to screwy behavior
 * if mapping feedback parameters on a note whose feedback is not
 * fixed length (e.g., loss of the note off). One workaround is to
 * use fixed length output; another is to manipulate feedback via
 * velocity.
 * (4) Possible additional trigger types: (1)  note/pitch value, (2) real-time
 * response to experimenter keystroke. Possible additional controls would 
 * be  (1) a flag for "affect one event only", (2) a flag for allowing 
 * control (for  each trigger type) of when the action takes place, (3) a
 * way of starting the count for a trigger on the basis of another
 * trigger (e.g., taps from start of continuation).
 * (5) Terminology: these used to be called "trigger events", or just "events".
 * "event" is a bit too generic, so some prefixes have been changed to "trig",
 * but many still are "EV" or "ev"
 * (6) Should trigger ID numbers be removed? They confuse
 * the process of making a file, and are a blatant source of user error.
 * The replacement capability for interactive mode is unimportant.
 * However, identification in the output file is useful.
 *
 * Modification history:
 */

#include <linux/types.h>
#include <errno.h>
#include <stdio.h>

#include "config.h"
#include "linux.h"
#include "ftaplimits.h"
#include "tapmidi.h"
#include "params.h"


/* structures containing all event types lumped together. This is the 
 * structure which contains the full descriptive information. May be empty,
 * or initialized by "EVENT" parameter lines.
 */
struct trig_event trig_event_table[MAX_TRIG_EVENTS] ;
int num_trig_events = 0 ;

/* structures for the ordered lists of metronome and keystroke events. 
 * Contains event time and index into trig_event_table
 */

struct cnt_event
{	int index ;
	int trig_cnt ;
} ;

struct cnt_event met_events[MAX_TRIG_EVENTS] ;
struct cnt_event key_events[MAX_TRIG_EVENTS] ;

int first_met_event, last_met_event ;
int first_key_event, last_key_event ;


/* ev_init(): do local initializations. The events have already been read into
 * the trig_event_table from the parameter file;  create the metronome and 
 * keystroke event lists from global trig_event_table[].  NOTE: I'm no longer
 * what will happen with, e.g., num_trig_events in interactive mode. So don't
 * do it for a real experiment!
 */

ev_init ()
{	
	event_list_init (TRIG_MET_TYPE, met_events, &first_met_event,
		&last_met_event) ;
	event_list_init (TRIG_KEY_TYPE, key_events, &first_key_event,
		&last_key_event) ;
#ifdef DEBUG
	printf("ev_init(): SORTED METRONOME EVENTS\n") ;
	print_event_list (last_met_event, met_events) ;
	printf("ev_init(): SORTED KEYSTROKE EVENTS\n") ;
	print_event_list (last_key_event, key_events) ;
#endif
}


/* ev_exec_action(): do the parameter change associated with the trigger 
 * event 
 */

ev_exec_action(index) 
int index;
{	struct trig_event *ep ;

#ifdef DEBUG
	printf("GOT TRIGGER %d\n", index) ;
#endif
	ep = &trig_event_table[index] ;
	if (ep->flags & END_EXP_FLAG) {
		end_experiment = 1 ;
		return ;
	}
	else if (ep->flags & FUNC_FLAG) {
		/* FIXME: more checks, encapsulation  */
		/* executue the function */
		(*(func_table[ep->val].func)) () ;
		return ;
	}
	else {
		*ep->paramp = ep->val ;
	}
		
}

/* ev_sched_time(): put all the time event triggers into the scheduling 
 * queue (which had better be active).  These triggers will be 
 * activated/checked from ProcOutput().
 */

ev_sched_time()
{	int i ;

	for (i = 0 ; i < num_trig_events ; i ++) {
		if (trig_event_table[i].type ==  TRIG_TIME_TYPE)  {
			sched_midi_note (0,  trig_event_table[i].id, i, 
				TRIG_TIME_TYPE, TE_TRIGGER, 
				trig_event_table[i].trig_cnt + trial_starttime) ;
		}
	}
}




/* 
 * Events are sorted both for efficiency, and for easy (logical)  removal of 
 * events once they've been executed. Possibly better done as a linked list.
 * This is a generic routine for keystroke and metronome events.
 */

event_list_init(flag, array, first, last) 
char flag ;
struct cnt_event array[] ;
int *first, *last ;
{
	int i ;
	struct trig_event *epp ;

	*first = 0 ;
	*last = 0 ;

	for (i = 0 ; i < num_trig_events ; i++) {
		if (trig_event_table[i].type == flag)  {
			array[*last].index = i ;
			array[*last].trig_cnt = 
				trig_event_table[i].trig_cnt ;
			(*last) ++ ;
		}
		
	}
	if (*last > 1)
		sort_events (*last,  array) ;
}

/* The following routines duplicate too much code, but i'm too lazy to put
 * together another parallel data structure.
 * Concept: met_events[] and key_events[] are sorted (low at bottom) lists
 * of events of those types. Once a trigger event has been acted upon, it's
 * finished, so effectively remove it by incrementing first_xxx_cnt. Because
 * of this, it's only necessary to look at the first event; callers of these
 * routines should loop until there are no more for the current count
 * Currently no allowance for missed events (== rather than >= ).
 */

is_metronome_event (count)
long count ;
{
	int i ;

	if ((i = first_met_event) < last_met_event) {  /* active events left */
		if (met_events[i].trig_cnt == count) {
			first_met_event++ ;
			return (met_events[i].index) ;
		}
	}
	return (-1) ;
}


is_key_event (count)
long count ;
{
	int i ;

	if ((i = first_key_event) < last_key_event) {  /* active events left */
		if (key_events[i].trig_cnt == count) {
			first_key_event++ ;
			return (key_events[i].index) ;
		}
	}
	return (-1) ;
}



/* both keyboard and metronome events must be sorted in ascending order.
 * 
 */

int event_cmp() ;

sort_events (n, events)
int n ;	
struct cnt_event events[] ;

{
 	/*  PORTNOTE: qsort(BA_LIB) may not be general. */
	qsort (&events[0], n, sizeof (struct cnt_event) , event_cmp) ;
}

event_cmp(event1, event2)
struct cnt_event *event1, *event2 ;
{	if (event1->trig_cnt < event2->trig_cnt)
		return (-1) ;
	else {
		if (event1->trig_cnt  == event2->trig_cnt)
			return(0) ;
		else
			return (1) ;
	}
}



#ifdef DEBUG

print_event_list(n, events)
int n ;
struct cnt_event events[] ;
{	int i ;
	for  (i = 0 ; i < n ; i++) {
		printf("entry %d: index %d trig_cnt %d\n",
			i, events[i].index, events[i].trig_cnt) ;
	}
}

#endif
