/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at sfinney@sfinney.com.
 */

/**************************************************************************
 * tapmidi.h: main include file for the ftap program.
 *
 * 3/12/00, sf, generalize mnote to handle controller events, though
 * these will _only_ undergo delay and channel mapping, and will
 * always be associated with FEED_CHAN1.
 * 3/13/00, sf, changed te_type into multiple subtypes, so that you only
 * need to look at the flag which is important. Do as bitfields. Output 
 * printing becomes more complicated.
 * 3/14/00, sf,  hopefully the final major hack, get rid of all the complications
 * of having a union, and just maintain status, byte1, and byte2 for
 * both notes and controllers. In fact, with the te_io flag it may be
 * superfluous to maintain both inevent and outevent structs, but
 * its usefulness in debugging probably outweighs any space concerns.
 * 6/25/01, sf, add a new trigger flag to signal that a function will
 * be called. The parameter value in the TRIGGER listings will choose
 * the function from a table. Note also TE_PULSE and bit field concerns.
 */



/**************************************************************************
 * The internal structure for storage of a MIDI "event", used for the output 
 * scheduling queue and for storage of MIDI in/out data. These events may be 
 * keystroke input, feedback, metronome, or a MIDI controller. In addition, 
 * trigger events are stored in this structure, in which case outnote.note 
 * will be the index into the event 
 * table (for internal use), and for printing purposes te_type will be set to 
 * 'T' (for event),  outnote.midichan will be used to indicate the trigger type 
 * ('K','M','T'), and outnote.vel will contain the (user-supplied) trigger 
 * event ID.
 *
 */

/* NOTE: triggers are encoded a little funny in this structure, with 
 * `midinote_onoff' storing the trigger type (T, M, K), `velocity' storing 
 * the external ID,  and `note' storing the internal ID. 
 */


/* (sub)structure for midi event data. Store it as basic MIDI message
 * format (although the status has had the channel stripped out),
 * but provide #defines for mnemonic note access 
 */

/* defines for midinote_onoff  in note_event structure */

/* MIDI (re)-defines for note on and note off (used internally) */

#define NOTEDOWN	0x90
#define NOTEUP		0x80

/* #corresponding values for file printout */
#define NOTEDOWN_CHAR	'D'
#define NOTEUP_CHAR	'U'

struct midi_event {
	unsigned long time ;	/* input/output time of midi event*/
	int  midichan ;

	unsigned char 	status;	/* status nibble only, channel removed */
#define midinote_onoff	status

/* 4/25/2016 sf hack: with more than 256 triggers, "byte1" has to 
 * be an int. Sigh.
 */
	unsigned int	byte1; 
#define midinote_pitch	byte1
  	
	unsigned char 	byte2 ;
#define midinote_vel	byte2
			
};

/* defines for te_type field in tap_event.  Different levels of
 * classification are now factored out. At printout time (write_data(),
 * plus any debug), these will get factored into the familiar "F", "K",
 * types.
 */

#define	TE_NOTE		0x01	/* a MIDI note */
#define TE_CONTROLLER	0x02	/* a MIDI controller */
#define TE_TRIGGER	0x04	/* an internal trigger */

/* Subtypes of TE_NOTE. For now, these will be set only if TE_NOTE is
 * set. FIXME: what are bit field limitations here? Adding PULSE
 */

#define TE_SUBTYPE_MASK 0xF0
#define TE_KEYSTROKE	0x10
#define TE_METRON	0x20
#define TE_JUNK		0x40  /* useless event for analysis (e.g., masking 
			       * noise onset) 
			       */
#define TE_PULSE	0x80

/* Input/output (may be  redundant with other defines). FIXME */
#define	TE_INPUT	0x01
#define	TE_OUTPUT	0x02


/* The main structure itself. 'innote' contains the time stamped input event.
 * 'outnote' will contain the event after any required mapping; this is what
 * will get written to MIDI output.
 */ 

struct tap_event {
	int  te_type ;	 
	char te_io ;
	struct midi_event inevent ;
	struct midi_event outevent ;
	
	int te_sequenceno ;	/* to detect reordering (obsolete?)  */
	int te_flags ;		/* for future use, NYI */

	/* single threaded (output) scheduling list */
	struct tap_event *te_p ;
};



struct tap_event * get_te() ;
struct tap_event * ReadMidiEvent() ;

extern struct tap_event te_schedlist ;

/* storage for all the keystroke pitches, indexed by sequenceno (for
 * lag pitch mapping).
 */

extern unsigned char keystroke_pitches[] ;

/****************************************************************************
 * Stuff relating to trigger events. A trigger event is defined by a particular
 * keystroke number, metronome beat number, or time, and causes an on-the-fly 
 * change in an integer-valued (only!) parameter.
 * an 
 */


/* Maximum size of a parameter name.
 */

#define MAX_PARAM 40

/* Values for type in (trigger) event structure */
#define TRIG_MET_TYPE	'M'
#define TRIG_KEY_TYPE	'K'
#define TRIG_TIME_TYPE	'T'

/* structure of (trigger) event table; see triggers.c.
 */

struct trig_event {
        char    type ; 
        int     id ;    
        int     flags ;		/* used  for  END_EXP_FLAG and FUNC_FLAG */
        int     trig_cnt ;      
	int	*paramp ;	/* value to be changed */
	char	paramnm[MAX_PARAM] ;	/* for printout */
        int	val ;		/* the new value */
} ;

extern struct trig_event trig_event_table[] ;

extern int num_trig_events ; 	/* number of event entries in event_table[] */



#define IS_TRIG_EVENT(X)   ((X == TRIG_MET_TYPE || X == TRIG_KEY_TYPE || X == TRIG_TIME_TYPE))

/* internally used  flag value, used to be only END_EXP. Things might screw up if the
 * this got changed to zero...
 */

#define END_EXP_FLAG	0x01
#define FUNC_FLAG	0x02

extern int end_experiment ;

/**************************************************************************
 * Important timing stuff.
 */


/* get the Linux time of day (time since 1/1/70 [?]) in milliseconds. Used
 * for scheduling and input time stamping, converted for input and output
 * to ms since trial start.
 */

unsigned long gettodms() ;


/* The following variable is crucial (and not well designed). Global time 
 * is from
 * the epoch (e.g., get_time_of_day() ), but output printout, click files, etc.,
 * are relative to the start of trial. trial_starttime is the global time of
 * trial start, and IT MUST BE SET BEFORE ROUTINES SUCH AS metron_start() are
 * called.
 */

extern unsigned long trial_starttime;   /* ms TOD when trial started */


#ifdef PPT

extern	int	ioi_cnt ;
extern  int	ioi_array[] ;

#endif 

/*************************************************************************
 * functions and triggers (see func_modules.c)
 */

extern struct func_desc {int index ; int (* func)() ; } func_table[] ;



/**************************************************************************
 * Miscellaneous stuff
 */


extern char filebase[] ;



extern char *paramfile, paramfilebuf[] ;


char *skip_token() ;


/* flag values for dump_params(): CURRENT for current values (for interactive
 * mode), INITIAL  for values before event changes (for file printout).
 */

#define CURRENT 1
#define INITIAL	2
