/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at sfinney@sfinney.com.
 */

/*
 * Modification History:
 * 6/28/04, sf: add FACTOR{N} parameters to encode conditions in the 
 * output file. They really should all be here, and not encoded in
 * the file name. The parameter file will encode some of these (in the
 * parameter file name, and redundantly but useful as parameters in the
 * file itself), and other more volative ones will be provided as arguments
 * at run time. Unfortunately, it's hard to anticipate how many
 * of these might be necessary; I'm adding more than I currently
 * need. 
 * Also, add a "SESSION" parameter; assumption is that subject +
 * session + block + trial is sufficent to uniquely identify a data
 * file. These values are used to construct the output file name. Again,
 * for some experiments this may not be sufficient, but it's more
 * flexible than the previous approach!@
 * 4/6/16, sf, add "mididev" parameter. Given how FTAP parses its
 *    command line, it's the easiest way to allow the user to override
 *    the default midi device (the old-school /dev/midi). Ideally we would
 *    confirm that the opened device is in fact an ALSA midi device.
 * TODO: provide an option to send redirect the file via stdout.
 */

/***************************************************************************
 * Routine containing initialization of configuration parameters, as well
 * as routines for reading them in from a file or command line. In general,
 * any experiment should be run from a file; the defaults here just allow
 * running a simple experiment without a file (i.e., just playing notes 
 * through (onto MIDI channel 1).
 *
 * Concept: all important trial characteristics (except for some level
 * of mapping control which must be C code) are defined by parameters. 
 * Parameters will typically be read from a parameter file. It is 
 * possible to override individual parameters from a command line 
 * argument, or in interactive mode; integer-valued parameters can also 
 * be changed on the fly _within a trial_ based on triggered events 
 * (see event.c). Many of these will do very useful things, some will
 * not. 
 *
 * There are 4 classes of parameters, each of which is handled by a somewhat
 * different data structure and routine. However, within each class additions
 * should be fairly easy. The types are:
 *  	1) Integer valued params, including on/off flags.
 *	2) String valued params (largely for documentation purposes, except
 *	 	for the newly-implemented click files)
 *	3) "Array parameters", i.e. more than one integer value, preceded by
 *		a count.
 *	4) Trigger-event descriptions; see below (special-cased).
 *
 * 
 * A parameter addition involves the necessary changes here, plus an
 * extern definition in params.h. For each class of parameter, there
 * is a table defining the I/O name (for reading in, and for printing, 
 * usually in capital letters), and the second is the address of the
 * variable to be set (the I/O name in lower case, plus "_param").
 * The tables are used both for parsing (set_params() and subroutines), and 
 * for printout (dump_params() ).
 *
 * NOTES:
 * (1) If interactive mode is being used, parameters are NOT reset to their 
 * initial values between runs; values changed by trigger events will stay 
 * changed.
 * (2) The relative automatization of this routine, and the ability to change
 * parameters on the fly, means that dependencies between parameters cannot
 * be checked. One such dependency is that the program will crash 
 * if RANDDELAY_ARRAY is 0, and you try to use random delays (DMODE = 2).
 *
 * SOME QUIRKS/LIMITATIONS
 * (1) Masking noise parameters cannot be effectively changed by triggers.
 * (2) Handling of stdout cannot be changed by triggers.
 * (3) If you have note off tracking the user's keystroke, and use triggers
 * to change feedN_chan, feedN_len, or some pitch mapping mode, and the trigger
 * occurs between a note down and a note up, you may not get proper NOTE_OFFs
 * sent. This should really be FIXME'd...
 * (4) There may be others...
 *
 *
 *
 * MODIFICATION HISTORY
 * 4/9/00, sf, revamp array params, adding metronome parameter _arrays_
 *	for vel, note, len, and midi channel, and going for a consistent
 *	naming convention with existing arrays. I considered making array
 *	length a separate (and trigger-able) integer parameter, but decided
 *	it was of little use and too much trouble.
 * 4/10/00, sf, add pitchseq_file param. Perhaps should be one for each
 *	FTAP channel. 
 * 8/10/00, sf, change the external name for the "FEED1_..." parameters 
 * 	to "FEED_"; this will be clearer for most users who won't care about 
 *	the second feedback channel. All internal code variables still use 
 * 	"feed1". 
 *	Also: the number of parameters has become unwieldy. Change the
 *	default printout  to be that only user-defined parameters, as well as
 *	some that I deem essential, are printed out. The entire set can
 *	be printed out for debugging purposes by setting FULL_PARAM_PRINT
 *	to 1. (Possibly the default should be that all params get printed
 *	out, and the user must turn the option off). "printit" set to 1 in
 *	a parameter table entry will force printout. Triggers will always be
 *	printed.
 * 8/11/00, sf, add a VMODE parameter so that velocity mappings can
 * 	be added in the future. Requires changing fchan_params as well.
 *  	Channel and note length will coded remain as 0 or fixed value. 
 * 6/25/01, sf, add PULSE params, and parsing of FUNC triggers. The immediate
 *	need to to use a trigger to sound a note (a "pulse"), but the function
 *	mechanism should be generally useful.
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "config.h"
#include "linux.h"
#include "ftaplimits.h"
#include "tapmidi.h"
#include "params.h"

/***********************************************************************
 * The array of addresses of values for the per-feedback channel parameters. 
 * This is a bit of extra indirection so that it's possible to have routines
 * which can access the current parameter values for either feedback channel,
 * even when those parameters have changed on the fly. Tightly coupled
 * to defines in params.h.
 */

int * fchan_params [NFCHANS] [NCHAN_PARAMS] = 
{ {  &feed1_on_param, &feed1_chan_param, &feed1_note_param, &feed1_vel_param,
     &feed1_len_param, &feed1_pmode_param,&feed1_dmode_param,&feed1_dval_param,
     &feed1_vmode_param},
  {  &feed2_on_param, &feed2_chan_param, &feed2_note_param, &feed2_vel_param,
     &feed2_len_param, &feed2_pmode_param,&feed2_dmode_param,&feed2_dval_param,
     &feed2_vmode_param}
} ;

/*************************************************************************
 * Data structures and defines for the different parameter types.
 **************************************************************************/


/************************************************************************
 * String-valued params, basically for documentation only (i.e., these
 * don't control anything about how the program actually runs, though
 * subject and trial are used to generate the output file name). For now,
 * only one comment line can be saved. Subj and trial cannot have embedded
 * spaces. The (single) comment line can have spaces.
 */

#define PARAM_STR_LEN	100

/* documentation parameters */
char mididev_param[PARAM_STR_LEN] = MIDIDEV;
char sub_param[PARAM_STR_LEN]	= "sub" ;
char session_param[PARAM_STR_LEN]	= "session" ;
char block_param[PARAM_STR_LEN]	= "block" ;
char trial_param[PARAM_STR_LEN]	= "trial" ;
char factor1_param[PARAM_STR_LEN]	= "factor" ;
char factor2_param[PARAM_STR_LEN]	= "factor" ;
char factor3_param[PARAM_STR_LEN]	= "factor" ;
char factor4_param[PARAM_STR_LEN]	= "factor" ;
char factor5_param[PARAM_STR_LEN]	= "factor" ;
char comment_param[PARAM_STR_LEN] = "" ;

/* output diagnostic parameters. These are not integer parameters,
 * as you don't want issues involving  current vs initial values!
 */

char sched_av_param[PARAM_STR_LEN] = " " ;
char sched_max_param[PARAM_STR_LEN] = " " ;
char sched_maxtime_param[PARAM_STR_LEN] = " " ;
char sched_gt1_param[PARAM_STR_LEN] = " " ;
char sched_gt5_param[PARAM_STR_LEN] = " " ;
char sched_gt10_param[PARAM_STR_LEN] = " " ;
char av_delay_param[PARAM_STR_LEN] = " " ;
char in_disc_max_param[PARAM_STR_LEN] = " " ;
char in_disc_max_time_param[PARAM_STR_LEN] = " " ;
char out_disc_av_param[PARAM_STR_LEN] = " " ;
char out_disc_max_param[PARAM_STR_LEN] = " " ;
char out_disc_max_time_param[PARAM_STR_LEN] = " " ;

/* names of (up to 2) external files specifying events to be output */
char click1_file_param[PARAM_STR_LEN] = "" ;
char click2_file_param[PARAM_STR_LEN] = "" ;

/* name of external files specifying events a sequence of pitch events to
 * be used for users keystrokes.
 */
char pitchseq_file_param[PARAM_STR_LEN] = "" ;

/* COMMENT is special-cased in parse_string to allow embedded spaces */
char comment_buf[100];		/* static storage for the (single) comment */

/* the structure defining string parameter names */
struct string_param
{	char *id ;
	char *valp ;
	int  printit ;		/* set to 1 forces printout */
}
string_params[] = 
{
{"MIDIDEV", 		mididev_param, 1},
{"SUB", 		sub_param, 1},
{"SESSION", 		session_param, 1},
{"BLOCK", 		block_param, 1},
{"TRIAL", 		trial_param, 1 },
{"FACTOR1", 		factor1_param, 1 },
{"FACTOR2", 		factor2_param, 1 },
{"FACTOR3", 		factor3_param, 1 },
{"FACTOR4", 		factor4_param, 1 },
{"FACTOR5", 		factor5_param, 1 },
{"COMMENT", 		comment_param, 0},
{"AV_DELAY", 		av_delay_param, 0},	/* output diagnostic only */
{"SCHED_AV", 		sched_av_param, 1},	/* output diagnostic only */
{"SCHED_MAX", 		sched_max_param, 1},	/* output diagnostic only */
{"SCHED_MAXTIME",	sched_maxtime_param, 1},/* output diagnostic only */
{"SCHED_GT1", 		sched_gt1_param, 1},	/* output diagnostic only */
{"SCHED_GT5", 		sched_gt5_param, 1},	/* output diagnostic only */
{"SCHED_GT10", 		sched_gt5_param, 1},	/* output diagnostic only */
{"IN_DISC_MAX", 	in_disc_max_param, 0},	/* output diagnostic only */
{"IN_DISC_MAX_TIME", 	in_disc_max_time_param, 0}, /* output diagnostic only */
{"OUT_DISC_AV", 	out_disc_av_param, 0},	/* output diagnostic only */
{"OUT_DISC_MAX", 	out_disc_max_param, 0},	/* output diagnostic only */
{"OUT_DISC_MAX_TIME", 	out_disc_max_time_param, 0},/* output diagnostic only */
{"CLICK1_FILE",		click1_file_param, 0},
{"CLICK2_FILE",		click2_file_param, 0},
{"PITCHSEQ_FILE",	pitchseq_file_param, 0},
{ (char *) NULL,	(char *) NULL} 
};

/**************************************************************************
 * Integer valued parameters (including on/off flags) 
 */

/********************************* 
 * Parameters relating to feedback from user's keystrokes, for each of two
 * feedback "channels". Current defaults: FEED1 is on and follows users
 * keystroke. Feed2 is off, but fixed length and delayed (like a click).
 */

int feed1_on_param = FEED_ALL ;
int feed1_chan_param = 1  ;
int feed1_note_param = 96 ;	/* Note value, if pmode = SAMEPITCH */
int feed1_vel_param = 0 ;	/* fixed MIDI velocity if non-zero,else input */
int feed1_len_param = 0  ;  	/* fixed note len if non-zero,else input */
int feed1_pmode_param = RIGHTPITCH;/* pitch mode mapping: default = input */
int feed1_dmode_param = SYNC_DELAY;/* delay mode mapping: default = sync  */
int feed1_dval_param = 250 ;	/* fixed delay in ms */
int feed1_vmode_param = 0 ;	/* if 0, copy input */

int feed2_on_param = FEED_OFF ;
int feed2_chan_param = 1  ;
int feed2_note_param = 80 ;	/* Note value, if pmode = SAMEPITCH */
int feed2_vel_param = 100 ;	/* fixed MIDI velocity if non-zero,else input */
int feed2_len_param = 20  ;  	/* fixed note len if non-zero,else input */
int feed2_pmode_param = SAMEPITCH;/* pitch mode mapping: default = same */
int feed2_dmode_param = FIXED_DELAY;/* delay mode mapping */
int feed2_dval_param = 250 ;	/* fixed delay in ms */
int feed2_vmode_param = 0 ;	/* if 0, copy input */

int split_point_param = 64 ;	/*If splitting is active, default to middle C*/

int pitchlag_param = 0 ;	/* play the current pitch */


/********************************* 
 * Parameters relating to masking noise: on at beginning of experiment, 
 * off at end.
 */

/* set to 1 to generate masking noise, 0 for no */
int mask_on_param = 0 ;
int mask_chan_param = 2 ;
int mask_note_param = 64  ;
int mask_vel_param = 35 ;


/********************************* 
 * Parameters relating to the metronome.
 * Metronome is fixed length, midi velocity, and pitch, although the rate
 * is varied by  mspb_param, and a rhythmic pattern can be imposed on it by
 * met_pattern_length_param. In addition, the integer parameters (such as
 * mspb, vel, and note) can  be changed by trigger events to create a 
 * time perturbation or a loudness or pitch change (perhaps these latter
 * two could be specified in a more complex 'pattern').  
 */


int met_chan_param = 1 ;
int met_note_param = 64;
int met_vel_param = 100;
int met_len_param = 20;		/* length of metronome tone  in milliseconds */


/* must be on for metronome to sound */
int metron_on_param = 0 ;

int mspb_param = 600;        /* ms per beat (100 bpm)  */

/*******************************************************************************
 *  parameters for programmed tones ("pulses")
 */

int pulse_vel_param = 100 ;
int pulse_chan_param = 1 ;
int pulse_note_param = 80 ;
int pulse_len_param = 1000 ;


/********************************* 
 * Miscellaneous integer parameters...
 */


/*
 * Parameter to control writing of output. If set to non-zero, write to 
 * stdout, not file  (in this case, only keystrokes will be written).
 * This should be set to 1 for practice sessions, so that internal data
 * structures don't overflow.
 */	
int stdout_param = 0 ;	


/* offset parameters for click files (so the same sequence can be
 * phase shifted and reused in different contexts)
 */
int click1_offset_param = 0 ;
int click2_offset_param = 0 ;


/* print out all parameters? For now, default to no. 
 */

int full_param_print_param =  0; 

#ifdef PPT

int delayppt_param = 50 ;
int delayppt_n_param = 1 ;
int delayppt_uplim_param = 1000 ;
int delayppt_lowlim_param = 30 ;

#endif

/********************************
 * The structure defining integer valued parameters
 */

struct int_param
{	char *id ;
	int *valp ;
	int init_val ;
	int printit ;		/* set to 1 forces printout */
}
int_params[] = 
{
{"FEED_ON", 		&feed1_on_param , -1, 1},
{"FEED_CHAN", 		&feed1_chan_param , -1, 1},
{"FEED_NOTE", 		&feed1_note_param , -1, 1},
{"FEED_VEL", 		&feed1_vel_param , -1, 1},
{"FEED_LEN", 		&feed1_len_param , -1, 1},
{"FEED_PMODE", 		&feed1_pmode_param , -1, 1},
{"FEED_DMODE", 		&feed1_dmode_param , -1, 1},
{"FEED_DVAL",		&feed1_dval_param , -1, 1},
{"FEED_VMODE",		&feed1_vmode_param , -1, 1},
{"FEED2_ON", 		&feed2_on_param , -1, 0},
{"FEED2_CHAN", 		&feed2_chan_param , -1, 0},
{"FEED2_NOTE", 		&feed2_note_param , -1, 0},
{"FEED2_VEL", 		&feed2_vel_param , -1, 0},
{"FEED2_LEN", 		&feed2_len_param , -1, 0},
{"FEED2_PMODE", 	&feed2_pmode_param , -1, 0},
{"FEED2_DMODE", 	&feed2_dmode_param , -1, 0},
{"FEED2_DVAL",		&feed2_dval_param , -1, 0},
{"FEED2_VMODE",		&feed2_vmode_param , -1, 0},
{"SPLIT_POINT",		&split_point_param, -1, 0},
{"PITCHLAG",		&pitchlag_param, -1, 0},
{"MASK_CHAN", 		&mask_chan_param , -1, 0},
{"MASK_NOTE", 		&mask_note_param , -1, 0},
{"MASK_VEL", 		&mask_vel_param , -1, 0},
{"MASK_ON", 		&mask_on_param , -1, 1},
{"MET_CHAN", 		&met_chan_param , -1, 1},
{"MET_NOTE", 		&met_note_param , -1, 1},
{"MET_VEL", 		&met_vel_param , -1, 1},
{"MET_LEN", 		&met_len_param , -1, 1},
{"METRON_ON", 		&metron_on_param , -1, 1},
{"MSPB", 		&mspb_param , -1, 1},
{"PULSE_CHAN", 		&pulse_chan_param , -1, 0},
{"PULSE_NOTE", 		&pulse_note_param , -1, 0},
{"PULSE_VEL", 		&pulse_vel_param , -1, 0},
{"PULSE_LEN", 		&pulse_len_param , -1, 0},
{"STDOUT", 		&stdout_param , -1, 0},
{"CLICK1_OFFSET", 	&click1_offset_param , -1, 0},
{"CLICK2_OFFSET", 	&click2_offset_param , -1, 0},
{"FULL_PARAM_PRINT", 	&full_param_print_param , -1, 1},
#ifdef PPT

{"DELAYPPT", 		&delayppt_param, -1, 0},
{"DELAYPPT_N", 		&delayppt_n_param , -1, 0},
{"DELAYPPT_UPLIM", 	&delayppt_uplim_param , -1, 0},
{"DELAYPPT_LOWLIM", 	&delayppt_lowlim_param , -1, 0},

#endif
{ (char *) NULL,	(int *) NULL, -1}
};

/*******************************************************************
 * Integer array valued parameters 
 */

/*
 *
 * Non-zero met_pattern_array_cnt will cause the metronome to go in a 
 * pattern defined by the first met_pattern_array_cnt_param fields of 
 * met_pattern_array_list_param[]  (where 1 is a sounding beat and 0 is a 
 * silent beat). 
 */  


int met_pattern_array_cnt_param  = 0;

int met_pattern_array_list_param[MAXPATTERN] = {} ;

/* arrays to pattern the sound quality of the metronome output. The counts
 * need not be equal (but usually will be).
 */

int met_note_array_cnt_param  = 0;

int met_note_array_list_param[MAXPATTERN] = {} ;

int met_vel_array_cnt_param  = 0;

int met_vel_array_list_param[MAXPATTERN] = {} ;

int met_len_array_cnt_param  = 0;

int met_len_array_list_param[MAXPATTERN] = {} ;

int met_chan_array_cnt_param  = 0;

int met_chan_array_list_param[MAXPATTERN] = {} ;

/*
 * Randomized delays: this allows random selection (per keystroke) of
 * a delay from a provided list of delay values. Other types of randomization
 * (e.g., using a normal or uniform distribution) will require C code
 * changes. Not parameterized per channel.
 */


/* number of values in random delay array */
int randdelay_array_cnt_param = 0 ;

/* the delay choices (ms)
 */

int randdelay_array_list_param[MAXRDELAYS] = {};


/********************************
 * The structure defining integer array valued parameters. Encoded in the
 * input file as <PARAM_NAME> <COUNT> <VALUES>...COUNT is checked against the
 * stored max value.
 */

struct array_param
{	char *id ;
	int max ;
	int *valp ;
	int *array ;
	int printit ;		/* set to 1 forces printout */
}
array_params[] = 
{
{"RANDDELAY_ARRAY", MAXRDELAYS,	&randdelay_array_cnt_param,
	randdelay_array_list_param, 0},
{"MET_PATTERN_ARRAY", MAXPATTERN, &met_pattern_array_cnt_param, 
	met_pattern_array_list_param, 0},
{"MET_VEL_ARRAY", MAXPATTERN, &met_vel_array_cnt_param, 
	met_vel_array_list_param, 0},
{"MET_NOTE_ARRAY", MAXPATTERN, &met_note_array_cnt_param, 
	met_note_array_list_param, 0},
{"MET_CHAN_ARRAY", MAXPATTERN, &met_chan_array_cnt_param, 
	met_chan_array_list_param, 0},
{"MET_LEN_ARRAY", MAXPATTERN, &met_len_array_cnt_param, 
	met_len_array_list_param, 0},
{ (char *) NULL, 0,(int *) NULL, (int *) NULL }
};

/*************************************************************************
 * Routines for parsing parameter files and lines.
 **************************************************************************/


/*************************************************************************
 * set_params(): read in and set whatever parameters the user has set in 
 * file "filename",  overriding the built in defaults.
 * 
 * File format: 0-length lines, or  lines starting with  white space or
 * "#", are ignored. Other lines are of the 4 formats described above. 
 * In general, just skip lines with errors, after printing a diagnostic.
 *
 * This routine calls a subroutine for parsing a single line, which
 * then calls subroutines for the different parameter types. 
 * Parameters must be parseable from file, command line, interactive, 
 * or trigger event actions.
 * 
 */


/* size limits for input line and parameter name */

#define MAX_LINE 256

int
set_params(filename)
char *filename ;
{	FILE *fp ;
	char buf[MAX_LINE + 1] ;

	num_trig_events = 0 ;	/* FIXME */
	if ((fp  = fopen (filename, "r")) == NULL) {
		printf("ERROR: can't open parameter file '%s'\n", filename) ;
		return (-1);
	}

	/* note: fgets will include the "\n" in buf (unlike gets() ) */
	while (fgets (buf, MAX_LINE, fp) != NULL) {
		if (strlen (buf) == 0 || buf[0] == ' ' || buf[0] == '\t' ||
			buf[0] == '\n' || buf[0] == '#')
			continue ;

		/* null terminate it:  FIXME (check) */
		buf[strlen(buf) - 1] = '\0' ;
		proc_param(buf) ;
	}
	fclose (fp) ;
	return (0) ;
}




/* process a single parameter data line. No checks for overflow. Get the
 * parameter name in one field, and the rest of the line for processing
 * by the subroutines; can't use scanf() because the rest of the line
 * may have spaces. The subroutines return 0 if the line wasn't processed
 * by that routine.
 */
proc_param(line)
char *line ;
{	char param[MAX_PARAM] ;
	char *remainder ;

#ifdef DEBUG
	printf("in proc_param, line = :%s:\n", line) ;
#endif

	/* get the parameter name */
	sscanf (line, "%s", param) ;

	/* get a pointer to  the arguments; there will be leading space */
	remainder = skip_token (line) ;

#ifdef DEBUG
	printf("in proc_param, param = :%s:  arg = :%s:\n", param, remainder) ;
#endif

	/* each routine will parse the params it knows about, else
	 * return 0.
	 */
	if (!parse_int_param(param, remainder))
	    if (!parse_string_param (param, remainder))
	        if (!parse_array_param (param, remainder))
		    if (!parse_trigger_param (param, remainder))
			printf ("proc_params: can't parse param: %s, %s\n",
				param, remainder ) ;
}





	
/* Separate parsing routines for each of the parameter types; a 
 * little ugly since much code (e.g., the search) is duplicated, but the
 * structures are different. Some error checking has been added, it may not
 * be sufficient.
 *
 * Note that the non-elegant code above this is such that "line" may
 * have leading spaces.
 *
 * The individual parse routines return 0 on error (which can mean either 
 * "entry not found" or "bad formatting": to do?), 1 on success.
 */

parse_string_param(name, line)
char *name, *line ;
{	struct string_param *ipp ;
	char *p ;
	int i ;

	i = 0 ;

	while (1) {
		p = string_params[i].id ;
		if (p == NULL)
			return (0) ;	/* not found */
		if (strcmp (p, name) == 0)
			break ;		/* found! */
		i++ ;
	}
	ipp = &string_params[i] ;
	/* special case COMMENT to allow embedded spaces */
	if (strcmp (name, "COMMENT") == 0) {
		strcpy (comment_buf, line) ;
		ipp->valp = comment_buf ;
		ipp->printit = 1 ;
		return (1) ;
	}
	if (sscanf(line, "%s", ipp->valp) != 1) {
		return (0) ;		/* failure for some reason */
	}
	ipp->printit = 1 ;

	return (1) ;			/* success */
}

parse_int_param(name, line)
char *name, *line ;
{	struct int_param *ipp ;
	char *p ;
	int i ;

	i = 0 ;

	while (1) {
		p = int_params[i].id ;
		if (p == NULL)
			return (0) ;	/* not found */
		if (strcmp (p, name) == 0)
			break ;		/* found! */
		i++ ;
	}
	ipp = &int_params[i] ;
	if (sscanf(line, "%d", ipp->valp) != 1) {
		return (0) ;		/* failure for some reason */
	}
	ipp->printit = 1 ;

	
	return (1) ;			/* success */
}


/* Could just include the values, without a leading count? */

parse_array_param (name, string)
char *name, *string ;
{
	int i, j, cnt;
	struct array_param *ipp ;
	int *ip ;
	char *p ;

	i = 0 ;

	while (1) {
		p = array_params[i].id ;
		if (p == NULL)
			return (0) ;	/* not found */
		if (strcmp (p, name) == 0)
			break ;		/* found! */
		i++ ;
	}
	ipp = &array_params[i] ;

	/* get the count (# of array entries) */
	if (sscanf(string, "%d", &cnt) != 1) {
		return (0) ;		/* failure for some reason */
	}
	if (cnt > ipp->max) {
		printf("Error: Param %s has more than > %d items\n", 
			name, ipp->max) ;
		return (0) ;
	}
	p = skip_token(string) ;

	* (ipp->valp) = cnt ;
 	ip = ipp->array ;

	for (j = 0 ; j < cnt; j++) {
		if (sscanf(p, "%d", ip + j) != 1) {
			return (0) ;
		} ;
		p = skip_token(p) ;
	}
	ipp->printit = 1 ;
	
}



/*
 * Routine to read event trigger.
 * File line format:
 *  TRIGGER <ID> <TYPE> <TRIG_CNT> <PARAM_NAME> <PARAM_VALUE>
 *
 *	TYPE is K(eyboard), M(etronome), or T(ime).
 *	ID is just a user ID number, for identifying file output lines, and
 *	also so that (by reusing an ID) you can overwrite an existing
 *	trigger.
 *	TRIG_CNT is # of down events, or millseconds for T.
 *	PARAM_NAME is the capital lettern name (only) of an integer (only!) 
 *	parameter, and PARAM_VALUE is the new value  (e.g. "FDELAY 100")
 *	For now, only such integer parameters can be used (and not all will
 *	work; see above). END_EXP is a special dummy parameter name which
 *	signals the end of the experiment.
 *
 * Internal array can be flexibly allocated to contain any types of
 * flags and combinations, although the total max number will be fixed.
 *
 * Note: although the primary intended use of this program is running from
 * a file, the interactive mode and parameter overrides suggest that it
 * ought to be possible to overwrite an existing trigger event. This is
 * done through the use of the ID field (which will also be written to the
 * output file). Although an event cannot be deleted, a new event with the
 * same ID will replace an old event with that ID, allowing changing of
 * a trigger and, in a kludgy way, deletion of an event by creating an
 * event which changes a parameter to be its existing value!
 * 
 * Reading in a parameter file will reset all event triggers.
 */


parse_trigger_param (name, string)
char *name, *string ;
{
	int i ;
	struct trig_event *ep, *ep2 ;
	int *ip ;
	char *p ;

	if (strcmp (name, "TRIGGER"))
		return (0) ;

	if (num_trig_events >= MAX_TRIG_EVENTS) {
		printf("Too many trigger events\n") ;
		return (0) ;
	}

#ifdef DEBUG
	printf("parse_event_param: string = :%s:\n", string) ;
#endif
	ep = &(trig_event_table[num_trig_events++]) ;
	if ((i = sscanf(string, "%d %c%d%s%d",&ep->id, &ep->type,
		&ep->trig_cnt, ep->paramnm, &ep->val )) != 5)  {
		num_trig_events-- ;
		return (0) ;
	}
	ep->flags = 0 ;		/* default value */

#ifdef DEBUG
	printf("parse_event_param: entry %d type %c trig_cnt %d\n",
		num_trig_events - 1, ep->type, ep->trig_cnt) ;
#endif

	if (!IS_TRIG_EVENT (ep->type)) {
		num_trig_events -- ;
		printf ("Bad trigger event type\n") ;
		return (0) ;
	}

	/* Check if there's an existing event with this ID (can't be more
	 * than one...)
	 */
	for (i = 0 ; i < num_trig_events - 1 ; i++) {
		if (trig_event_table[i].id == ep->id) {  /* replacement */
			printf("WARNING: overwriting trigger event %d\n",
				ep->id) ;
			ep2 = &(trig_event_table[i]) ;
			ep2->type = ep->type ;
			ep2->flags = ep->flags ;
			ep2->trig_cnt = ep->trig_cnt ;
			strcpy (ep2->paramnm, ep->paramnm) ;
			ep2->val = ep->val ;
			ep = ep2 ;	/* for rest of function */
			num_trig_events-- ;
		}
	}

	/* special case the end experiment parameter. It's identified in 
	 * the parameter file by a name, but it's implemented as a
	 * flag value 
	 */
	if (!strcmp (ep->paramnm, "END_EXP")) {
		ep->flags |= END_EXP_FLAG ;
		return (1) ;
	}
	/* special case the  FUNC parameter. Also implemented by a flag
	 * value, but the parameter value is used to code the function (magic!)
	 * That value was set above.
	 */
	if (!strcmp (ep->paramnm, "FUNC")) {
		ep->flags |= FUNC_FLAG;
		return (1) ;
	}
	/* look up the integer parameter name */
	i = 0 ;
	while (1) {
		p = int_params[i].id ;
		if (p == NULL) {
			num_trig_events-- ;
			printf ("Event: no such integer parameter\n") ;
			return (0) ;	/* not found */
		}
		if (strcmp (p, ep->paramnm) == 0)
			break ;		/* found! */
		i++ ;
	}
	/* copy the address of the actual C variable to be set */
	ep->paramp = int_params[i].valp ;
	return (1) ;

	
}


/***************************************************************************
 * Printout routines
 */

extern char *paramfile; 
extern char version_number[] ;
const char build_time[] = __DATE__ " " __TIME__;

/* 'flag' determines whether the current parameter values should be
 * printed (as in interactive mode), or the initial parameter values
 * (for documentation in an output file in normal running mode). The
 * only difference is integer parameters, which may have been changed
 * by triggers.
 */

extern int running_as_root;

dump_params(fp, flag)
FILE *fp ;
{	
	char *p ;
	fprintf(fp,"# VERSION_NUMBER\t%s\n", version_number) ;
	fprintf(fp, "# PARAMETER_FILE\t%s\n", 
		paramfile == NULL ? "NONE" : paramfile) ;
	fprintf(fp, "# PERMISSIONS\t%s\n", 
		running_as_root? "root" : "user") ;
        fprintf(fp, "# BUILD_TIME\t%s\n", build_time);
	print_string_params(fp) ;
	if (flag == CURRENT)
		print_current_int_params(fp) ;
	else if (flag == INITIAL)
		print_initial_int_params(fp) ;
	print_array_params(fp) ;
	print_event_params(fp) ;
}

print_string_params(fp)
FILE *fp ;
{

	int i ;

	i = 0 ;
	while (1) {
		if (string_params[i].id == NULL)
			break ;
		if (full_param_print_param || string_params[i].printit == 1)
			fprintf(fp, "# %s\t\t\t%s\n", string_params[i].id, 
				string_params[i].valp) ;	
		i++ ;
	}
}

/* routine to save all the initial parameter values in the init_val 
 * field.
 */

save_int_params(fp)
FILE *fp ;
{

	int i ;

	i = 0 ;
	while (1) {
		if (int_params[i].id == NULL)
			break ;
		int_params[i].init_val = *(int_params[i].valp) ;	
		i++ ;
	}
}

/* print out the _initial_ values of the integer parameters... */

print_initial_int_params(fp)
FILE *fp ;
{

	int i ;

	i = 0 ;
	while (1) {
		if (int_params[i].id == NULL)
			break ;
		if (full_param_print_param || int_params[i].printit == 1)
			fprintf(fp, "# %s\t\t%d\n", int_params[i].id, 
				int_params[i].init_val) ;	
		i++ ;
	}
}

/* print out the _current_ values of the integer parameters... */

print_current_int_params(fp)
FILE *fp ;
{

	int i ;

	i = 0 ;
	while (1) {
		if (int_params[i].id == NULL)
			break ;
		if (full_param_print_param || int_params[i].printit == 1)
			fprintf(fp, "# %s\t\t%d\n", int_params[i].id, 
				*(int_params[i].valp)) ;	
		i++ ;
	}
}


print_array_params (fp) 
FILE *fp  ;
{
	int i, j, cnt;
	int *ip ;

	i = 0 ;
	while (1) {
		if (array_params[i].id == NULL)
			break ;
		if (full_param_print_param == 0  && array_params[i].printit == 0) {
			i++ ;
			continue; 
		}
		cnt = * (array_params[i].valp) ;
		ip = array_params[i].array ;
		fprintf(fp, "# %s\t\t%d", array_params[i].id, 
			cnt) ;	
		for (j = 0 ; j < cnt; j++) {
			fprintf(fp, "\t%d", *(ip + j)) ;
		}
		fprintf(fp, "\n") ;
		i++ ;
	}
}

print_event_params(fp)
FILE *fp ;
{
	int i;
	struct trig_event *ep ;

	for (i = 0 ; i < num_trig_events ; i++) {
		ep = &trig_event_table[i] ;
		fprintf(fp, "# %s %d %c %d %s %d\n",  "TRIGGER", ep->id,
			ep->type, ep->trig_cnt, ep->paramnm, 
			ep->val) ;

	}

}


/* return a pointer to the first (whitespace) character following 
 * the first found token. The resulting string will thus start with
 * leading whitespace.
 * Could use better error checking?
 */

char *skip_token(s)
char *s ;
{	
	char *s1 = s ;	/* for debugging */
	while (isspace (*s1++)) {} ;
	while (!isspace (*s1++)) {} ;

	return (s1) ;
}
