/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at sfinney@sfinney.com.
 */


/********************************************************************
 * Output routines:
 * The central routine here  (ProcOutput() ) will be called once 
 * a millisecond (or more often) from the main routine.  The main routine will 
 * also call InitOutput() and EndOutput() routines as necessary.
 *
 * Tapevents for output are place in a scheduling queue, which is processed at
 * least  once a millisecond (we hope!); output events scheduled for that 
 * time (or earlier) are set to MIDI out. All output events (feedback and 
 * metronome) go into the same queue. When an output event is written, it 
 * is also stored in the mdoutstore[] array for later printing (if 
 * stdout_param is set, there is no storage  or printing of output events; 
 * only keystroke events are written; this allows for  practice runs with 
 * no risk of overflowing data structures).
 *
 * Time and metronome trigger events are detected and executed
 * here (keystroke events are executed elsewhere, immediately when the
 * keystroke is read).
 *
 * The metronome is partially  implemented here (see also metron.c).
 * At the (relative) top of sched() call check_metron(), which will put 
 * metronome events in the output queue if there are any to be scheduled.
 * 
 * 
 * Note: there used to be code in here which kept track of missequencing of
 * out events (e.g, for random delay), but it was bogus if feedback was off
 * for part of the trial, so it has been removed. The data is available in
 * the output file anyway.
 *
 * NOTE: the gt1, gt5, gt10 output diagnostics are the key diagnostic to
 * whether FTAP's timing is adequate. The input and ouput scheduling
 * variables merely allow looking at exactly where something went wrong
 * in cases where the file might be suspect. In fact, perhaps they should
 * be removed. 
 *
 ************************************************************************
 * Functions provided to outside world:
 *	ProcOutput()
 *	InitOutput()
 * 	EndOutput()
 *   
 *
 * Modification history:
 *	3/00: sf, major Linux hack.
 *	3/20/00, sf, metronome triggers are now processed  in check_metron 
 * 	  code; they won't be in output queue. Time triggers are still in 
 *        output queue, so they don't have to be checked every single 
 *	  scheduling cycle.
 *	
 */



#include <linux/types.h>
#include <signal.h>
#include <stdio.h>
#include <errno.h>

#include "config.h"
#include "linux.h"
#include "ftaplimits.h"
#include "tapmidi.h"
#include "params.h"

extern int errno ;



/**************************************************************************
 * statistics/program behavior variables.
 */


/* variables to keep track of how often ProcOutput is being called. This was more
 * important when scheduling was interrupt driven, but still may be useful.
 */
unsigned long	lasttime ;

int     prof_schedtot , prof_schedcnt , gt1cnt, gt5cnt, gt10cnt, firstsched = 1,
	max_sched_sz, max_sched_cnt, max_sched_offtime ;

unsigned int max_sched_epochtime ;

/* variables to keep track of the time between when an event is supposed to
 * go out, and when it actually goes out. This will serve as a diagnostic of
 * how efficiently this program is working; however, it CANNOT diagnose delays
 * in the Linux driver itself (which is a genuine concern, as some drivers
 * only do 10 ms granularity output polling). An external test (ftaploop) will
 * be necessary to test this.
 */

int     sched_event_cnt, sched_event_disc_tot, sched_event_disc_max,
                sched_event_disc_max_time ;

extern int sched_input_max, sched_input_max_time ;




/* Storage for output events;  the same sort of structures as used for internal 
 * control 
 */

struct tap_event mdoutstore[MAXNOTES] ;
int		mdoutstore_index ;



/* the routine which will be called (hopefully) at least once a ms to
 * schedule output.
 * 
 * Note: sequenceno values keep track of down events, to track output
 * reordering when random delay is used. te_sequenceno is 0 for note-up
 * events. This _assumes_ that the only events we'll see here with sequence
 * numbers are feedback for key down events.
 */

ProcOutput()
{	
	
	unsigned long curtime, diff, tmp ;
	int  retval;

	char updown ;

	struct tap_event *locte_p, *tmp_p, *startte_p  ;

	struct tap_event *mdop ;

	curtime = gettodms() ;

	/* count how often scheduling occurs */
	if (firstsched) {
		lasttime = curtime ;
		prof_schedtot = 0 ;
		prof_schedcnt = 0 ;
		gt1cnt = 0 ;
		gt5cnt = 0 ;
		gt10cnt = 0 ;
		firstsched = 0 ;
		max_sched_sz = 0 ;
		max_sched_cnt = 0 ;
		max_sched_offtime = 0 ;
		max_sched_epochtime = 0 ;
		sched_input_max = 0 ; 
		sched_input_max_time = 0  ;
                sched_event_cnt = 0 ;
                sched_event_disc_tot = 0 ;
                sched_event_disc_max = 0 ;
                sched_event_disc_max_time = 0 ;

	}
	else {
		diff = curtime - lasttime ;
		prof_schedtot += diff ;
		if (diff > 1)
			gt1cnt++ ;
		if (diff > 5)
			gt5cnt++ ;
		if (diff > 10)
			gt10cnt++ ;
		prof_schedcnt += 1 ;
		if (diff > max_sched_sz) {
			max_sched_sz = diff ;
			max_sched_cnt = prof_schedcnt ;
			max_sched_epochtime = curtime ;
			max_sched_offtime = curtime - trial_starttime ;
		}
		lasttime = curtime ;
	
	}


#ifdef DEBUG
        {
        long tmptime = 0 ;
        printf("in ProcOutput\n") ;
        count_queues() ;
        if (te_schedlist.te_p != NULL)
                tmptime = te_schedlist.te_p->outevent.time ;



        printf("curtime %lu, 1st queue time %lu \n", 
		curtime, tmptime) ;
        }
#endif


	check_metron(curtime) ;

        while ((locte_p = te_schedlist.te_p) != NULL &&
                        locte_p->outevent.time <= curtime) { 


#ifdef DEBUG
	printf("output loop: locte_p = 0x%lx, locte_p->te_p 0x%lx\n",(unsigned long)locte_p, (unsigned long)locte_p->te_p) ;
#endif


		/* if not writing to std_out (which is a mode for 
		 * practice/experimentation/debugging) save the message 
		 * (whether MIDI information, or a trigger event).
		 */
		if (!stdout_param) {
			mdop = & (mdoutstore[mdoutstore_index]) ;
                	if (++mdoutstore_index > MAXNOTES){
                        	printf("ERROR: MIDIOUTSTORE OVERFLOW!!\n");
				mdoutstore_index -- ;
                	}
			else {
				copy_te (locte_p, mdop) ;
                        	/* write the ACTUAL output scheduling time, not
                         	* just the desired one.
                         	*/
                        	mdop->outevent.time = curtime ;
			}

		}
#ifdef DEBUG
		printf("before Trigger Check%lu\n", gettodms() ) ;
#endif


		/* check for dummy (trigger event) messages. Any trigger 
		 * event encountered here should be executed immediately.
 		 * outevent.note has been set up as an index into the event 
		 * table.
		 * The only thing that matters here is that something is a 
		 * trigger event, not its subtype...
		 */
		if (locte_p->te_type & TE_TRIGGER) {
			ev_exec_action (locte_p->outevent.midinote_vel) ;

			/* free it */
                        tmp_p = te_schedlist.te_p ;
                        te_schedlist.te_p = tmp_p->te_p ;

			free_te(tmp_p) ;
			continue ;
		}
#ifdef DEBUG
		printf("after Trigger Check%lu\n", gettodms() ) ;
#endif


                /* It's not a trigger event....
 		 *
		 * Record scheduling discrepancy for non-triggers,
                 * skipping scheduling times of 0 (these used to be  sometimes
                 * used, e.g. for masks, for immediate scheduling). Note that 
                 * (fixed) curtime is used, so if there's a huge number of 
		 * simultaneous output events, this will not be precisely 
		 * accurate (this should not  occur in any real-life situation)
		 * This diagnostic does not take into account any delay within 
		 * the MIDI interface itself.
                 */
                if (tmp = locte_p->outevent.time)  {
                        tmp = curtime - tmp ;         /* diff */
                        sched_event_cnt ++ ;
                        sched_event_disc_tot += tmp ;
                        if (tmp > sched_event_disc_max) {
                                sched_event_disc_max = tmp ;
                                sched_event_disc_max_time = curtime - trial_starttime ;
                        }

                }


		/* Write the MIDI event to the output port.
		 * Should really need to check for errors here.
		 */
#ifdef DEBUG
		printf("before WriteMidiEvent %lu\n", gettodms() ) ;
		PrintTapEvent (locte_p) ;
#endif

		WriteMidiEvent(locte_p) ;
#ifdef DEBUG
		printf("after WriteMidiEvent %lu\n", gettodms() ) ;
#endif


		/* put the sent event back on the free list */
                tmp_p = te_schedlist.te_p ;
                te_schedlist.te_p = tmp_p->te_p ;

		free_te(tmp_p) ;
	}
}


/* set up some output variables.
 */

InitOutput()
{


	firstsched = 1 ; /* reset counters */
	mdoutstore_index = 0 ;

}



/* finish up with some output variables
 */

EndOutput()
{
	
	float av_sched_time, av_disc_time ;
	FlushOutput() ;



	av_sched_time = (float) prof_schedtot/prof_schedcnt ;
        av_disc_time = (float) sched_event_disc_tot/sched_event_cnt;
        /* Print the  on the fly diagnostic to std out */
        printf("Mean time between sched()'s (ms): %.2f, schedcnt: %d \n"
		"> 1 ms: %d, > 5 ms: %d, > 10 ms: %d, max: %d ms\n",
                av_sched_time, prof_schedcnt, gt1cnt, gt5cnt, gt10cnt,
                max_sched_sz) ;
        /* stash variables for printout as parameters. These values may
         * be bogus if both (a) writing to std out, and (b) doing multiple
         * trials in one tap session. Big deal.
         */
        sprintf (sched_av_param, "%.2f", av_sched_time) ;
        sprintf (sched_max_param, "%d", max_sched_sz) ;
	sprintf (sched_maxtime_param, "%d", max_sched_offtime) ;
	sprintf (sched_gt1_param, "%d", gt1cnt) ;
	sprintf (sched_gt5_param, "%d", gt5cnt) ;
	sprintf (sched_gt10_param, "%d", gt10cnt) ;
        sprintf (in_disc_max_param, "%d", sched_input_max) ;
        sprintf (in_disc_max_time_param, "%d", sched_input_max_time) ;
        sprintf (out_disc_av_param, "%.2f", av_disc_time) ;
        sprintf (out_disc_max_param, "%d", sched_event_disc_max) ;
        sprintf (out_disc_max_time_param, "%d", sched_event_disc_max_time) ;
}

/* This code is called when the experiment ends. Allow about 5-10 ms
 * to flush pending events, and then make sure all notes get turned off.
 */


FlushOutput()
{	int i ;
	for (i = 0 ; i < 10 ; i++) {
		ProcOutput() ;
		RTCNap() ;
	}
	MidiNotesOff() ;
}



