/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at sfinney@sfinney.com.
 */


/***********************************************************************
 * Isolate the code which does pitch (and other) mappings. 
 * These routines are generally freely applicable to 
 * either of the two feedback channels, though random delay and the
 * random and sequence pitch mappings are not completely independent (and
 * have not been thoroughly thought through). Eventually these should be 
 * applicable to the metronome as well.
 * 
 * Mapping routines fall into two categories. (1) Midi channel allows
 * only a very simple mapping: if the controlling parameter is 0, copy the 
 * input value, else use the fixed value that the parameter is set to.
 * (2) Delay, pitch, and velocity allow a wider variety of mappings. 0
 * means ``no change from input", but other values allow various capabilities,
 * sometimes invoking other parameter values.
 *
 * Fixed length note vs following subject's key off (feedN_len
 * parameter) is handled elsewhere.
 *
 * Any additional mappings will require C-code change, but at least the
 * framework is provided here. If a desired pitch or delay mapping is not
 * provided by  the current setup, additional cases could be added to 
 * delaymap() or pitchmap(), along with additional functions following the 
 * models * here.
 *
 * Various things complicate coding here. There are multiple feedback
 * channels. Parameters may be changed on the fly by trigger events. For 
 * random delay, you presumably want the key up to receive the same delay
 * as the corresponding key down. For some of the pitch mappings, you need
 * to make sure that the key up triggers a NOTEOFF for the same note sent
 * on the key down. I have attempted to handle all of these things, but
 * not all combinations will necessarily work (e.g., trying to independently
 * assign the same pitch sequence mapping to the two channels).
 *
 * Also, if the tone generator only allows a single instance of a note to
 * be active at a time, there may be some problems  (e.g., note clippings
 * in seq2). A better "same note without clipping" could be implemented by 
 * a same-note tuned tone generator, with (perhaps) a scalar tone output.
 *
 * Do you need separate keydown storage arrays for each pitch mapping
 * function?  (e.g., seq1save[])
 *
 *
 * Wish List:
 *       Make "channels" (particularly split keyboard) more general; it'd be
 *       nice to have separate pitch manipulations on the two sides of the
 *       split, particularly for pitch sequences and pitch lag. Unfortunately,
 *       pitch lag is tricky, because you need to _store_ the data separately
 *       for the two portions of the keyboard!
 *
 * Make an explicit mechanism for pitch sequences having silent events?
 * 
 * It'd be kind of fun to be able to make key releases trigger a NOTE ON,
 * but this is likely to be difficult (and a key press must send a NOTEOFF
 * for a different note value?)
 *
 * Modification History:
 *	3/11/00, sf, messages may now be controllers, such as sustain
 *	pedal (or pitch bend, or ..). These will go through delay and
 * 	midi channel mapping, but _not_ through pitch or velocity mapping
 *	(see ftap.c/map_note() ). 
 *	3/20/00, sf, add pitchlag pitch mapping: if pmode is LAGPITCH, then
 *	use the pitch from pitchlag keystrokes back. If there haven't been
 *	enough keystrokes, then use the fixed_note value. Note that this
 *	old _pitch_ value will be used with the current keystrokes output
 *	values (velocity and timing). Better be FEED1 channel.
 *	4/10/00, sf, replace BACHPITCH and SCALEPITCH with more general
 * 	FILESEQPITCH. Currently only one; should probably be per feedback
 *	channel, along with LAGPITCH. Latter is hard, as it requires 
 *	separation on _input_.
 *	8/11/00, sf, add a VMODE parameter, for future velocity extensions.
 *
 *	4/18/01, sf, major hack to handle feedback parameter-changing triggers
 *	which occur between NOTEON and NOTEOFF: this was handled correctly
 *	for delay, and had also been noted as a problem for FEED_ON, but it
 *	also turns out to be a problem for changing pitch mode, MIDI channel,
 *	or length! So, _always_ record the mapping (for all values) for a key
 *	press, and _always_ retrieve those values for the corresponding
 *	key release. Implicitly assumed is (1) a given pressed key will always
 *	produce a NOTEOFF before a second NOTEON, and (2) no keys are pressed
 *	when the trial starts.
 *	Note: (1) the delaymap and chanmap routines apply to controllers as
 *	well as keypresses (pitchmap and velmap only apply to keystrokes).
 *	Note: (2) the mods here will cause check_soundnote to return
 *	0 for the NOTEOFF of a fixed-length feedback tone (the NOTEOFF has
 *	already been scheduled); this means some of the code and comments
 *	in the main loop in ftap.c is no longer needed (and may be misleading);
 *	however, everything should work.
 *	Note: (3) Velocity does _not_ have to be saved, since its value
 *	in a NOTEOFF message is irrelevant.
 *	Note: (4) Although I don't see (at the moment) any particular use
 *	for mapping of length, the combined encoding of mapping and value 
 *	in the FEED_LEN parameter is ugly.
 *	9/6/01, check_soundnote fix (folded in from 2.1.05).
 *
 *	
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "linux.h"
#include "params.h"
#include "tapmidi.h"

/* presumably any note (MIDI key number) will be less than the following 
 * value.
 */

#define MAXKEYS	150

/* The new, big, global "saved keypress values" array. Complicated because
 * the two feedback channels may behave differently! NOTE: although the
 * keypressed flag is logically _shared_ across the two feedback channels,
 * it's very hard to track it that way. check_soundnote will always
 * get called for both feedback channels, so include it in the per-channel
 * structure. It should stay consistent.
 */

struct keypress
{	
	/* send_noteoff is a flag, 1=yes, 0=no. Both FEED_LEN and FEED_ON are
	 * encoded: if, at the time of key press, FEED_LEN is non-zero, or
	 * FEED_ON is 0, then no NOTEOFF is required at key release   
 	 * 
	 * keypressed is a consistency check: 1 if a key is currently depressed
	 * (dysthymia doesn't count :-) )
	 */
	int	keypressed, /* flag: 0/1, assumes 0 (global) initialization */
		send_noteoff,  
		mchan,		/* MIDI channel */
		note,
		delay;
} ;

/* storage for all notes for the two feedback channels */
struct  keypress saved_keyvals[2][MAXKEYS] ;
	

/* values to calculate actual random delay (sanity check in random
 * delay case).
 */

int delay_cnt ;
int delay_tot ;

/***********************************************************************
 * routine to determine if a note sounds at all: return 1 if yes, 0 if
 * no sound. 
 */

check_soundnote(fchan, tep)
int fchan ;
struct tap_event *tep ;
{	int  mode, note, soundflag ;
	
	mode = * (fchan_params [fchan] [FCHAN_ON_INDEX]) ;

	/* Controller weirdness: for a controller event, this will only
	 * get called on FCHAN1. Don't worry about splits for controllers.
	 */
	if (tep->te_type & TE_CONTROLLER)  {
		if (mode == FEED_OFF)
			return (0) ;
		else
			return (1) ;
	};



	note = tep->inevent.midinote_pitch ;	/* the keypress value */

	/* assume it's a keystroke now... */
	if ( tep->inevent.midinote_onoff == NOTEDOWN)  {

#ifdef DEBUG_KP
		printf("C_S: note down, note = %d, chan = %d\n", note, fchan) ;

#endif
		/* paranoia check */
		if (saved_keyvals[fchan][note].keypressed)
			printf ("ERROR: check_soundnote, keypress = 1!\n") ;

		saved_keyvals[fchan][note].keypressed = 1 ;
		switch (mode)
		{
		 	case FEED_OFF:	soundflag = 0 ;
					break ;
	
			case FEED_ALL:	soundflag = 1 ;
					break ;
	
			case FEED_ABOVE:if (note >= split_point_param)
						soundflag = 1 ;
					else
						soundflag = 0 ;
					break ;

			case FEED_BELOW:if (note <  split_point_param)
						soundflag = 1 ;
					else
						soundflag = 0 ;
					break ;
		}
		/* If FEED_ON is 0, or it's a fixed length feedback tone, the
 		 * key release should not trigger any feedback.
		 */
		if (soundflag & (*fchan_params[fchan][FCHAN_LEN_INDEX] == 0))
			saved_keyvals[fchan][note].send_noteoff = 1 ;
		else
			saved_keyvals[fchan][note].send_noteoff = 0 ;
#ifdef DEBUG_KP
			printf("soundflag = %d\n", soundflag) ;
#endif
		return (soundflag) ;
	}
	else {		/* NOTE UP */

#ifdef DEBUG_KP
		printf("C_S: note up, note = %d, chan = %d\n", note, fchan) ;
		printf("keypressed %d, send_noteoff %d\n", 
			saved_keyvals[fchan][note].keypressed,
			saved_keyvals[fchan][note].send_noteoff);
#endif
		if (saved_keyvals[fchan][note].keypressed == 0) 
			printf("ERROR: check_soundnote: no keypress!!\n") ;
		saved_keyvals[fchan][note].keypressed = 0 ;
		return (saved_keyvals[fchan][note].send_noteoff);
	}
	/* should never reach here */
	printf("ERROR: check_soundnote: shouldn't reach here!\n") ;
}



/************************************************************************
 * DELAY-mapping code
 *
 * Note: No longer maintaining statistics for random delay (issue with
 * multiple channels).
 * This routine may contain excess work for the simplest (synchronous)
 * case, given that this code is somewhat time-critical.
 * There are  different fixed delay values per channel; random delay is 
 * not variable across channels.
 * 
 * NOTE: controllers may be mapped through here (for consistency), but
 * random delay behavior is not guaranteed!
 */


/* hard wired lower bound and distribution width for a uniform random 
 * distribution. Can be changed; could be parameterized if it's important.
 */

#define UNIFORM_LOWER  100
#define UNIFORM_UPPER  300

delaymap(fchan, tep)
int fchan ;
struct tap_event *tep ;
{	int delay, mode ;
	
	mode = * (fchan_params [fchan] [FCHAN_DMODE_INDEX]) ;

	/* initialize to the fixed delay value */
	delay = * (fchan_params [fchan] [FCHAN_DVAL_INDEX]);

	/* if a NOTEUP, get the delay for the preceding corresponding
	 * NOTEDOWN.
 	 */
	if (tep->te_type & TE_KEYSTROKE && 
	   tep->inevent.midinote_onoff == NOTEUP)  {
		delay = saved_keyvals[fchan][tep->inevent.midinote_pitch].delay ;
	}
	else {
		switch (mode)
		{	case SYNC_DELAY:	delay = 0 ;
						break ;
	
			case FIXED_DELAY:	/* delay was set above */
						break ;
	
			case RAND_DELAY: 	delay = 
				randdelay_array_list_param[getrand (0,
					randdelay_array_cnt_param - 1)] ;
						break ;

			case UNIFORM_DELAY:	delay = getrand (UNIFORM_LOWER,
						   UNIFORM_UPPER);
						break ;


#ifdef PPT
			case PPT_DELAY:		if (ioi_cnt != 0)
							delay = pptdelay() ;
						/* else use the default
						 * fixed value.
						 */
							
						break ;
#endif

			

			default:  printf("delaymap(): unimplemented mode: %d\n",
					mode) ;
				  exit(-1);

		}

		/* save delay from a note down event */
		if (tep->te_type & TE_KEYSTROKE) {
			saved_keyvals[fchan][tep->inevent.midinote_pitch].delay 
				= delay ;
		}

		/* maintain stats for delays (note down events only here) */
		delay_tot += delay ;
		delay_cnt += 1 ;
	}
	tep->outevent.time = tep->inevent.time + delay ;
}

#ifdef PPT

/* Return a delay value based on a proportion of the subject's IOI's (up to
 * the last delayppt_n of them). It's always correct to do the calculation from
 * the start of the circular buffer.
 * 
 * NOTE: average is done as integer arithmetic, which means rounding
 * will always be done _down_. delayppt_param is an integer
 * proportion.
 *
 * NOTE:
 * For now, calculated anew each time, but could be done by maintaining
 * a running average and adding and subtracting the appropriate amount.
 * But, do you have the old value? 
 *
 * NOTE: this routine should be called only if ioi_cnt != 0.
 *
 */

int 
pptdelay()
{	int i, cnt, delay ;
	long total = 0 ;


	if (ioi_cnt > delayppt_n_param) 
		cnt = delayppt_n_param ;
	else
		cnt = ioi_cnt ;

	for (i = 0 ; i < cnt ; i++) 
		total += ioi_array[i] ;
	delay =  (total * delayppt_param)/(cnt * 100) ;
#ifdef PPTDEBUG
	printf("in pptdelay(), cnt = %d, delay = %d\n", delayppt_n_param, delay) ;
#endif
	return (delay) ;
}


#endif 

/************************************************************************
 * PITCH-mapping code
 * RANDPITCH: for comparability in an experiment, you may want to compare 
 * RANDPITCH to a SAMEPITCH, and have notes be in the same range. So _if_ 
 * FEED1_NOTE is non-zero, randomize around it (earlier versions would 
 * ignore FEED1_NOTE in all cases except SAMEPITCH)
 *
 * LAGPITCH cannot handle multiple channels, but it will use
 * whatever channel is supplied here rather than defaulting to FEED1_CHAN!
 * 
 * Routines here involve both (1) keyboard mapping (changing
 * a particular key on the keyboard to sound a particular note), and
 * (2) keystroke mapping.
 */


pitchmap (fchan, tep) 
int fchan;
struct tap_event *tep ;
{
	int outnote, innote, mode, tmpnote ;
	int fixed_note ;
	int tmp_sequenceno ;
	unsigned char onoff ;

	mode = * (fchan_params [fchan] [FCHAN_PMODE_INDEX]) ;
	fixed_note = *(fchan_params[fchan] [FCHAN_NOTE_INDEX]);

	innote = tep->inevent.midinote_pitch;
	onoff = tep->inevent.midinote_onoff ;

	if (tep->te_type & TE_KEYSTROKE && onoff == NOTEUP)  {
		outnote = saved_keyvals[fchan][innote].note;
	}
	else {
		switch (mode)		
		{
			case RIGHTPITCH:outnote = innote;
					break ;
	
			case SAMEPITCH: outnote = fixed_note;
					break ;
	
			case REVPITCH: 	outnote = revpitch (innote)  ;
					break ;
	
			case LARGEPITCH:outnote = largepitch (innote)  ;
					break ;
	
				/* RANDPITCH varies around fixed_note if it's
				 * set, else the keystroke value 
				 */
			case RANDPITCH:	if (fixed_note)
						tmpnote = fixed_note ;
					else
						tmpnote = innote ;
					outnote=randpitch(tmpnote,  onoff) ;
					break ;
	
			case FILESEQPITCH: outnote= seqpitch(innote, onoff);
					break ;
	
		
			case LAGPITCH:	outnote = lagpitch (innote, 
						tep->te_sequenceno, fixed_note, 
						onoff) ;
					break ;
	
	/*
			case USER_PITCH_START: outnote=user1pitch(innote, onoff);
					 break ;
	 */
	
			default: printf("pitchmap(): unimplemented mode: %d\n",
						mode) ;
				 exit(-1);
	
		}
		/* save mapped pitch from a note down event */
		if (tep->te_type & TE_KEYSTROKE) {
			saved_keyvals[fchan][innote].note = outnote ;
		}
	}
	tep->outevent.midinote_pitch = outnote ;
}


/* REVPITCH: Reverse the keyboard.
 * Note: TOPNOTE may have some dependency on the particular keyboard/tone
 * generator (as may any conception of middle C).
 */

#define TOPNOTE 120

revpitch (innote)
int innote ;
{
        return (TOPNOTE - innote) ;

}


/***********************************************************************
 * LARGEPITCH:  Give an arbitrary pitch mapping, but one which is consistent, 
 * and repeats at the octave. This is the arbitrary "large pitch" mapping of 
 * earlier experiments. In fact, it looks as if that program neglected to 
 * map pitches A, Bb, and B (bug!)
 * 0 is C (as is 60 and 120!).
 */

#define NNOTES 12	/* Notes in an octave */

int  large_map[NNOTES] = {+6, -2, +3, 0, +15, -6, 0, +1, -5, +1, +2, 0} ;

largepitch (innote)
int innote ;
{
        return (innote + large_map[innote % NNOTES]) ;

}

/***************************************************************************
 * RANDPITCH:  map each keystroke within the range of a fourth below (7 half 
 * steps, see getrand() call) and a fourth  above, randomly per keystroke. Use 
 * the same random number generator  as used in choosing random delays. 
 * Include one check so the same note doesn't get sent out  twice in a row, 
 * to avoid one obvious overlap.
 *
 * Note: both random pitch and sequence pitch require saving the
 * (unpredictable) mapping on a down note so that when the corresponding
 * up event comes in it can be properly mapped. In fact, there may be
 * a problem if the pmode were to change in the middle: so it better not!
 * Maintain separate map arrays for each, in case separate feedback 
 * channels are assigned to each. 
 */


#define P_RANGE 7		/* half-steps in a 4th */

int randnotes[MAXKEYS] ;

int rand_lastnote ;

randpitch (innote, updown)
int innote ;
unsigned char updown ;	/* should be int? FIXME, throughout */
{
	int note ;

	if (updown == NOTEUP) {
		return (randnotes[innote]) ;
	}
	else { /* NOTEDOWN */
		note = innote + getrand (-P_RANGE,  +P_RANGE) ;
		if (note == rand_lastnote)
			note += 1 ;
		rand_lastnote = note ;
		randnotes[innote] = note ;
		return (note) ;
	}

}

/* FILESEQPITCH:  generate output notes from a fixed input pitch sequence, 
 * regardless of what the input note is. Since
 * these play notes from a fixed sequence, starting indices need to be 
 * initialized before rerunning (reset_indexes). 
 * The sequences used to be hard-wired, but are now read in from
 * a file. Currently only one; probably should have one per channel.  FIXME.
 * NOTE: entries in file cannot have leading space...FIXME.
 *
 * Note that counting through the sequence implicitly occurs only when
 * feedback is turned _ON_ (because this routine is only called under
 * that circumstance).
 */

/* maximum length of a user-specified pitch sequence */
#define MAX_PITCHSEQ	500

int pitchseq_index = 0 ;	/* current position when playing back. */

/* the sequence of pitches, and the number of entries */
int pitchseq_list[MAX_PITCHSEQ] ;
int pitchseq_cnt = 0 ;


/* values for last mapping of a down event */
int pitchseq_save[MAXKEYS] ;


seqpitch (innote, updown)
int innote ;
unsigned char updown ;	
{
	int note ;

	if (pitchseq_cnt  == 0) {
		printf("pitchseq mapping: no entries!\n") ;
		exit (-1) ;
	}

	if (updown == NOTEUP) {
		return (pitchseq_save[innote]) ;
	} 
	else { /* NOTEDOWN */
		note = pitchseq_list[pitchseq_index] ;
		pitchseq_save[innote] = note ;
		if (++pitchseq_index >= pitchseq_cnt)
			pitchseq_index = 0 ;	/* reset when at end */
		return (note) ;

	}

}

/* dummy/filler for a user pitch mapping routine */

user1pitch (innote, updown)
int innote ;
unsigned char updown ;	
{
	printf("Unimplemented USER_PITCH_START mapping") ;
	return (0) ;
}

/**************************************************************************
 * LAGPITCH:  generate output pitches from pitchlag keystrokes back.
 */


/* values for last mapping of a down event (FIXME: must this be a new
 * variable? 
 */

int lagsave[MAXKEYS] ;



/* NOTE: what happens if FEED1_ON gets changed?? */

int lagpitch (innote, sequence, fixed_note,  updown)
int innote, sequence, fixed_note ;
unsigned char updown ;	
{
	int note, tmp_sequenceno ;

	if (updown == NOTEUP) {
		return (lagsave[innote]) ;
	} 
	else { /* NOTEDOWN */
		tmp_sequenceno = sequence - pitchlag_param ;
		if (tmp_sequenceno < 1)	/* 0 is not a sequence number */
			note = fixed_note ;
		else
			note = keystroke_pitches[tmp_sequenceno] ;
		lagsave[innote] = note ;
	}

	return (note) ;

}

/*************************************************************************
 * Simple velocity and midi channel mappings. REV_VEL and RAND_VEL are somewhat
 * arbitrary: what approximately "works".
 * REV_VEL: reverse the velocity (thus, setting loudness _against_ pressure).
 *    MIDI velocity ranges from 0 to 128.
 * RAN_VEL: randomize velocity between 30 and 100 (arbitrary).
 *
 * By fiat, no velocity mapping on NOTEOFF events (and input code, also
 * by fiat, has set velocity in such events to 0). Thus, it is not currently
 * possible to make the key press send a NOTE_OFF message.
 
 * Code elsewhere has made sure this is called only on KEYSTROKE EVENTS

 * This may become extended some day (at one point, I thought I had a use 
 * for a constant added or subtracted from the key velocity.
 */

velmap (fchan, tep) 
int fchan ;
struct tap_event *tep ;
{

	int vel_param, vmode ;
	int val ;
	
	vmode = * (fchan_params [fchan] [FCHAN_VMODE_INDEX]) ;

	vel_param = * (fchan_params [fchan] [FCHAN_VEL_INDEX]);

		
	if (vmode == KEY_VEL || tep->inevent.midinote_onoff == NOTEUP)
		tep->outevent.midinote_vel = 
			tep->inevent.midinote_vel ;
	else
	switch (vmode) 
	{
		case FIXED_VEL: tep->outevent.midinote_vel = vel_param ;
				break ;

		case REV_VEL: 	val = 160 - tep->inevent.midinote_vel;
				if (val > 127)
					val = 127 ;
				tep->outevent.midinote_vel = val;
	
				break ;

		case RAND_VEL: 	tep->outevent.midinote_vel = 
				35 + 20 *  getrand (0, 4) ;
				break ;

		default:	printf("velmap(): unimplemented mode: %d\n",
					vel_param) ;
				exit(-1);
	}
	
}

/* note: MIDI channels are defined from 1-16 (even though internal
 * message value for 1 is 0). For consistency with above,allow either 
 * a fixed user defined output channel, or copy the input channel, the
 * latter signalled by the parameter value being 0.
 */

chanmap (fchan, tep) 
int fchan ;
struct tap_event *tep ;
{
	int chan_param, mchan ;

	chan_param = * (fchan_params [fchan] [FCHAN_CHAN_INDEX]) ;

	/* if a NOTEUP, get the channel for the preceding corresponding
	 * NOTEDOWN.
 	 */
	if (tep->te_type & TE_KEYSTROKE && 
	   tep->inevent.midinote_onoff == NOTEUP)  {
		mchan = saved_keyvals[fchan][tep->inevent.midinote_pitch].mchan ;
	}
	else {
		if (chan_param)
			mchan = chan_param ;
		else
			mchan = tep->inevent.midichan ;

		/* save midi channel from a note down event */
		if (tep->te_type & TE_KEYSTROKE) {
			saved_keyvals[fchan][tep->inevent.midinote_pitch].mchan 
				= mchan ;
		}
	}

	tep->outevent.midichan = mchan ;
}

/* Utility routine to zero all indexes and other mapping-related variables
 * before rerunning 
 */
reset_pitch_indexes()
{
	pitchseq_index = 0 ;
	rand_lastnote = 0 ;
}

/* Read in the pitchseq file */

/* size limit for input line */
#define MAX_LINE 100

ReadPitchSeqFile(infilename) 
char *infilename ;
{
	FILE *infile ;
	char buf[MAX_LINE + 1] ;
	int i ;
	

	pitchseq_cnt = 0 ;

	infile = fopen (infilename, "r") ;
	if (infile == NULL) {
		printf("ReadPitchSequenceFile: cannot open input file %s\n", 
			infilename) ;
		exit (-1) ;
	}
	/* note: fgets will include the "\n" in buf (unlike gets() ) */
	while (fgets (buf, MAX_LINE, infile) != NULL) {
		if (strlen (buf) == 0 || buf[0] == ' ' || buf[0] == '\t' ||
			buf[0] == '\n' || buf[0] == '#')
			continue ;
		i = atoi (buf) ;
		if (pitchseq_cnt < MAX_PITCHSEQ - 1) { /* off by one? FIXME */
			/* no note range check */
			pitchseq_list[pitchseq_cnt++] = i ;
		}
		else {
			printf("More than %d items in %s; truncating\n",
				MAX_PITCHSEQ, infilename) ; /* off by one? */ 
			break ;
		}
	}
	fclose(infile) ;
}
