/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at sfinney@sfinney.com.
 */


/* MODIFICATION HISTORY:
 * 6/25/01, M000, sf. The addition of (the pulse) function called on triggers 
 * caused a bug: the
 * insert event had a >= which caused an inserted event to be placed _before_
 * another event with the same time in the scheduling queue. A time trigger
 * was at the head of the queue, and the pulse on insertion screwed up
 * pointers. Change the >= to > (see below). This shouldn't have any
 * negative consequences. Proper mutual exclusion on queue access would
 * have prevented this bug (but possibly caused deadlock!).
 */


/************************************************************************
 * Utility routines for the tap_event structures, including free and 
 * scheduling list management. The tap_event structure is the 
 * currency for all internal midi communication (i.e., everything except 
 * immediate input and output). 
 *
 * NOTE: these structures must _not_ be manipulated asynchronously (i.e.,
 * at interrupts); the list manipulations are not protected. The scheduling 
 * and free lists are just single-threaded.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "linux.h"
#include "tapmidi.h"
#include "ftaplimits.h"


/* dummy list heads for free list and scheduled events, + actual
 * data storage. 
 */

struct tap_event te_freelist, te_schedlist, te_array [MAX_TE_EVENTS] ;

void 
init_te_queue() 
{
	int i ;
	
	te_schedlist.te_p = NULL ;
	for (i = 0 ; i < MAX_TE_EVENTS - 1 ;i++) {
		te_array[i].te_p = &te_array[i+1] ;
	}
	te_freelist.te_p = &te_array[0] ;
	te_array[MAX_TE_EVENTS - 1].te_p = NULL ;
}

/* get a tap_event  from front of free list 
 */
struct tap_event *
get_te()
{
	struct tap_event *tmp_p ;

	if (te_freelist.te_p != NULL) {
		tmp_p = te_freelist.te_p ;
		te_freelist.te_p = tmp_p->te_p ;
		/* FIXME: do other initializations? */
		tmp_p->te_flags = 0 ;
		tmp_p->te_type = 0 ;	/* this initialization is important */
		tmp_p->te_io = 0 ;
		return (tmp_p) ;
	}
	/* big time error..*/
	printf("ERROR: get_te: no more events!\n") ;
	exit (-1);
}

/* free to front of free list 
 */
free_te(free_p)
struct tap_event *free_p ;
{
	free_p->te_p = te_freelist.te_p ;
	te_freelist.te_p = free_p ;
}


/* Place a tap_event in correct timing location in the scheduling list. Assume
 * that list access is safe, either because the final design only has
 * one piece of code accessing it, or because it is semaphore protected.
 * NOTE: implicitly assuming 32 bit integers (ms time as an integer).
 */

insert_event (new_p) 
struct tap_event *new_p ;
{
	struct tap_event *lastp, *nextp ;
	int etime ;

	etime = new_p->outevent.time;
	lastp = &te_schedlist ;
	nextp = te_schedlist.te_p ;

#ifdef DEBUG
	printf("insert_event():") ;
	PrintTapEvent(new_p)  ;
#endif

	while (nextp != NULL) {
		/* M000, change >= to >. The return will cause only one insertion
		 * to occur */
		if (nextp->outevent.time > etime) { 
			lastp->te_p = new_p ;
			new_p->te_p = nextp ;
			return ;
		}
		lastp = nextp ;
		nextp = nextp->te_p ;
	}
	/* end of list */
	lastp->te_p = new_p ;
	new_p->te_p = NULL ;
}

/* schedule an internally generated event...for now, only filling in
 * output fields. This may be wrong for (future) perturbed metronome, which 
 * will need to be mapped. Events created here will be metronome events
 * and trigger events; in the latter case, some fields get special use 
 * 
 * NOTE/FIXME: these are assumed to always be note (or trigger) events, 
 * not controller events.
 *
 * Inconsistent use of integer type?
 */

sched_midi_note (chan, note, velocity, onoff, type, time)
int chan, time, type ;
unsigned char note, velocity, onoff ;
{	
	struct tap_event *tep ;

	tep = get_te() ;
	tep->outevent.time = time ;
	tep->outevent.midichan = chan ;
	tep->outevent.midinote_pitch = note ;
	tep->outevent.midinote_vel = velocity ;
	tep->outevent.midinote_onoff = onoff;
	tep->te_type = type ;  
	tep->te_io = TE_OUTPUT ;
	tep->te_sequenceno = 0 ;	/* 0 for metronome events (??) */
	insert_event(tep) ;
}


/**********************************************************************
 * Utility routines.
 */

copy_te(src, dest)
struct tap_event *src, *dest ;
{	
	memcpy ( (char *) dest, (char *) src, sizeof (struct tap_event));
	
}


/***********************************************************************
 * debugging routines
 */


print_sched()
{	struct tap_event *p1, *p2 ;
	int i, j ;

	p1 = te_schedlist.te_p ;
	i = 0 ;

	while (p1 != NULL) {
		i ++ ;
		printf ("item %d time %lu\n", i, p1->outevent.time) ;
		p1 = p1->te_p ;
	}
	printf("End of schedule list: %d items\n", i) ;
}

count_queues()
{	struct tap_event *p1;
	int schedcnt = 0, freecnt = 0 ;

	for (p1 = te_schedlist.te_p ; p1 != NULL; p1 = p1->te_p) 
		schedcnt++;
	for (p1 = te_freelist.te_p ; p1 != NULL; p1 = p1->te_p) 
		freecnt++;
	printf("schedlist %d items, freelist %d items\n", schedcnt,
		freecnt) ;
}


