/************************************************************************
 * playftap: a Linux-based, MIDI-based program for "playing" (that
 * is, given an auditory representation for) an FTAP output file.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at finney.17@osu.edu.
 */

/***********************************************************************
  
   playtap.c: This is a playback program for FTAP output files, making
   it possible to listen to the data files (including
   things one didn't hear when running the experiment, like the keystrokes
   themselves!).  Various aspects of what you hear can be specified by
   command line arguments. It currently requires that the input file have
   the .abs suffix created by FTAP.  It defaults to playing keystrokes if 
   nothing else is specified.

   This code has not been kept up to date, but should work. The accuracy
   of the program argument descriptions needs to be verified.

   USAGE: there are a bunch of command line arguments which allow choosing 
	exactly what to listen to. It will not be possible to listen to "junk" 
	events (masking noise), however, you may want to listen to trigger 
	events (to know where things changed). You certain want to listen to 
	keystroke events, which on the original run did not (in and of 
	themselves) make any noise. This means that it may be necessary to
	allow specifying MIDI channel for keystokes, since the input keystroke
	value may not be suitable for the tone generator.

	playftap usage is as follows, where all arguments (except filename)
	are optional.  This program currently only plays one file at
	a time.

	playtap -kmft -R<ratio> -T<tchan> -U<tnote> -C<keystrokechan>
		-K<keystrokeadd> -S<startpoint> filename

	The "k", "m", "f", and "t" turn on playback of keystrokes, metronome, 
	feedback, and trigger events respectively; you can play these 
	simultaneously in any combination. Since trigger events in the
	file  don't have a MIDI channel or note, the "T" and "U" arguments
	allow overriding the built-in defaults of 1 and 120 for channel and 
	note, respectively. (The sounding time of 75 ms and velocity of
	100 for trigger playback are hardwired).
	Since keyboard midi channel input doesn't necessarily correspond to a 
	reasonable sound, "C" allows overriding the MIDI channel in the
	input file (perhaps a note should be added, at least
	for tapping (vs musical) experiments), and "K" provides a value you
	can add to keystroke velocity values (to make them louder); its use
	automatically turns on keystroke audio ouptut. The "S" flag
	allows for a millisecond time (from file start) for when playback 
	of the file should begin. The "R" flag allows for altering the
	rate of playback (slowing down in sometimes useful); 2.0 will play
	at half speed, .5 will play at double speed.
	
  
   Design: Read the input file, write everything to a scheduling queue, and
  	then run output from that queue. This just seems the simplest.
   Concerns:
	(1) Because of the design, the active events maximum (MAX_TE_EVENTS
	in list.c) becomes the TOTAL events maximum.
	(2) Because of including tapmidi.h but NOT including all of the FTAP
	modules, it is necessary to define some unavailable references for
	the linker,  even though they're not used.
	(3) It's necessary to write a stripped-down simple_sched(), since 
	all the bells are whistles of the ftap.c one get in the way.
 	(4) This should probably run at real time (i.e., suid root) for 
	accurate output scheduling. This will eat up the CPU time in a loop, 
	so put a pause() in the loop.
	(5) Currently, the metronome and (fixed) feedback are configured
	louder than the (non-sounded!) keystrokes, so use ?VELDEC for
	adjustment.
 	(6) NOTE: currently ignores anything except F(eedback), K(eystroke),
	M(etronome) and T(rigger) events (e.g., no controllers).
	(7) Certain aspects of this derive from my own experimental
        setup. For example, I program metronome and feedback to be
	fairly loud in my experiments, so MVELDEC and FVELDEC are
	built-in velocity subtractions from metronome and feedback
	output, respectively.

   Possible to do: 
   	: default to skip silence at beginning of file.
        : handle controllers properly 
        : Need to test again 
   	: put in "no data to play" diagnostic
        : add a flag to print out the time every second or 
		so (to be able to go back and  look at something interesting). 
		However, this requires jumping into the loop somehow...

   This file links with a locally modified version of lists.c (to get 
   increased limits from a local ftaplimits.h), as well as with linux_midi.c 
   and linux_utils.c from the main FTAP directory. This file itself is a 
   heavily modified and simplified version of chunks of the main FTAP code.
  
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <sys/types.h>
#include <sys/times.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <setjmp.h>
#include <errno.h>

#include "../../src/tapmidi.h"


unsigned long trial_starttime ;    /* TOD (in ms) when MIDI ports opened, 
				 * for scheduling 
				 */


/********************************************************************
 * file stuff 
 */

/* output file */
#define FILE_SUF ".abs"




/********************************************************************
 * General/miscellaneous stuff
 */

char *p ;

char buf[100], *bp  ;


int	end_experiment ;

/********************************************************************
 *  PORTNOTE: UNIX stuff 
 */

extern int errno ;
jmp_buf term_buf ;
void sig_int() ;
void init_realtime() ;

/***********************************************************************
 *  CONFIGURABLE PARAMETERS; eventually will be command line arguments 
 */

/* flags for sounding feedback, keystroke, metronome, and trigger
 * events
 */
int 	fflag = 0,
	kflag = 0,
	mflag = 0,
	tflag = 0;

/* default choices for keystroke MIDI channel, and sounds
 * for trigger events (may be parameterized later). Note that the input
 * channel for a keystroke even is not necessarily the same way you want
 * it to sound! Trigger events don't have any plausibly associated sound.
 */

int	kchan = 1,
	tchan = 1,
	tnote = 120,
	kadd = 0 ; 

double   ratio = 1.0 ;	 /* 2.0 will mean multiply all times by two, that is,
			  * slow down the playback. atof() requires a double.
			  */

/* startpoint must default to 0; setting it negative may have weird effects */
/* filter out any events occuring before this */
int 	startpoint = 0 ;   

#define TLEN	75	/* length of sound for trigger event */
#define TVEL	100	/* velocity for sounded trigger event */

/* the metronome and feedback are configured to be fairly loud; decrement 
 * the velocity 
 * so the user's keystrokes are more obvious. FIXME, this is somewhat 
 * specific to my
 * current choices of loudness, and the fact that feedback has a fixed velocity.
 */ 
#define MVELDEC	20
#define FVELDEC	20

main(argc, argv)
char **argv ;
{	int i, j;


	
	char 	*infilename ;


	/* Getopt stuff (my first use!) */
	extern	int optind, opterr ;
	extern  char *optarg ;

	int 	opt, errflag = 0 ;
	
	opterr = 0 ;
		
	while ((opt  = getopt (argc, argv, "kfmtpR:K:C:T:U:S:")) != EOF)
		switch (opt) 
		{
			case 'k':	kflag = 1 ;
					break ;

			case 'f':	fflag = 1 ;
					break ;

			case 'm':	mflag = 1 ;
					break ;

			case 't':	tflag = 1 ;
					break ;

			case 'R':	ratio = atof (optarg) ;
					break ;

			case 'K':	kadd = atoi (optarg) ;
					kflag = 1 ;
					break ;

			case 'C':	kchan = atoi (optarg) ;
					break ;

			case 'T':	tchan = atoi (optarg) ;
					break ;

			case 'U':	tnote = atoi (optarg) ;
					break ;

			case 'S':	startpoint = atoi (optarg) ;
					break ;

			default:	errflag = 1 ;
					break ;
		}

	if (errflag || optind >= argc) {
		printf("Usage: playtap -kmft -R<ratio> -T<tchan>" 
		 "  -U<tnote> -K<keychan> \n\t\t-S<startpoint> filename\n" ) ;
		exit (-1) ;
	}
	/* if user hasn't specified anything, default to playing keystrokes */
	if ((kflag  + fflag + mflag) == 0) 
		kflag = 1 ;
	infilename = argv[optind] ;



	/* one-time startup stuff */
	InitRealTime () ;
	InitMidi() ;




	/* a bunch of per-trial/experiment initialization */

	init_te_queue() ;	/* just reset all scheduling queues */
	InitSaveBuf(); /* for making sure all notes get turned off */


	/* write all the (chosen) events in the input file to the
	 * scheduling queue.
	 */
	setup_sched(infilename) ;


	printf("hit INTR (ctl-c) to stop....\n") ;


	/* PORTNOTE: UNIX */
	signal(SIGINT, sig_int) ;

	/* PORTNOTE: SGI */
	trial_starttime = gettodms() ;

	StartRTC() ;


	/* exiting from the experiment may be based on setting of the flag
	 * end_experiment, which can happen with a user ctrl-C
 	 * interrupt
	 */

	while (!end_experiment && te_schedlist.te_p != NULL) {
		RTCNap() ;   /* will stick in a .49 ms pause if
			      * running as root (with high scheduling
			      * priority) to allow interrupts.
			      */
		simple_sched() ;
	}


	exit(0) ;
}


/***********************************************************************
 * Read the input file, writing all the selected data to the scheduling
 * queue. For feedback and metronome events, this is straightforward. For
 * keystroke events, it may be necessary to map the midi channel. For
 * trigger events (sounded so that the listener can know when changes
 * occur), it's necessary to fix a midi channel and note, and create
 * an off event (for now, arbitrary defined as 50 ms after the on event).
 */


/* size limit for input line */
#define MAX_LINE 100

setup_sched(infilename) 
char *infilename ;
{
	FILE *infile ;
	char buf[MAX_LINE + 1] ;
	struct tap_event *te_outp ;	
	int i ;
	static  lasttrigtime = 0 ;
	
	/* fields for data lines in input file... */
	int time, midichan, notenum, velocity, sequence ;

	/* do as strings, since scanf and char may have probs: FIXME */
	char 	notename[10], updown[10], typestr[10] ;

	char type ;

	infile = fopen (infilename, "r") ;
	if (infile == NULL) {
		printf("Error: cannot open input file %s\n", infilename) ;
		exit (-1) ;
	}
	/* note: fgets will include the "\n" in buf (unlike gets() ) */
	while (fgets (buf, MAX_LINE, infile) != NULL) {
		if (strlen (buf) == 0 || buf[0] == ' ' || buf[0] == '\t' ||
			buf[0] == '\n' || buf[0] == '#')
			continue ;

		i = sscanf (buf, "%d%s%d%d%s%d%d%s", &time, updown, &midichan,
			&notenum, notename, &velocity, &sequence, typestr ) ;
		type = typestr[0] ;
		if (i != 8) {
                	buf[strlen(buf) - 1] = '\0' ;
			printf("sscanf error %d: %s\n", i, buf) ;
			exit(-1) ;
		}
		if (time < startpoint)	/* filter based on start time */
			continue ;
		/* adjust time so that we don't have silence for the duration */
		time = time - startpoint ;
		if (type == 'J')
			continue ;	/* masking noise, ignore */
		if ((type == 'F' && fflag) || (type == 'M' && mflag) ||
			(type == 'K' && kflag)) {
			te_outp = get_te() ;
			te_outp->outevent.time = (int) (ratio * time) ;
			te_outp->outevent.midinote_pitch = notenum ;
			/* arbitrary adjustment of velocities so that 
			 * keystrokes (the component of most variety and
			 * interest) are more audible. Maybe FIXME.
 			 */
			if ((type == 'F') && (velocity > FVELDEC)) {
				te_outp->outevent.midinote_vel = 
					velocity - FVELDEC ;
			}
			else {
			   if ((type == 'M') && (velocity > MVELDEC))
		   		te_outp->outevent.midinote_vel = 
					velocity - MVELDEC ;
			    else
				te_outp->outevent.midinote_vel = velocity ;
			}
			if (updown[0] == NOTEDOWN_CHAR)
				te_outp->outevent.midinote_onoff = NOTEDOWN ;
			else if (updown[0] == NOTEUP_CHAR)
				te_outp->outevent.midinote_onoff = NOTEUP ;
			else {
				printf("unparseable noteup/down char '%d'\n",
					updown[0]) ;
				exit (-1) ;
			}
		
			if (type == 'K') {
				te_outp->outevent.midichan = kchan ;
				te_outp->outevent.midinote_pitch += kadd ;
			}
			else
				te_outp->outevent.midichan = midichan ;
			insert_event (te_outp) ;
		} 
		/* sound for a trigger event, if desired */
		if (type == 'E' && tflag) {
			/* Note on */
			te_outp = get_te() ;
			te_outp->outevent.time = (int) (ratio * time) ;
			te_outp->outevent.midinote_pitch = tnote ;
			te_outp->outevent.midinote_vel = TVEL ;
			te_outp->outevent.midinote_onoff = NOTEDOWN;
			te_outp->outevent.midichan = tchan ;
			insert_event (te_outp) ;
			/* Note off */
			te_outp = get_te() ;
			te_outp->outevent.time = (int) (ratio * (time + TLEN)) ;
			te_outp->outevent.midinote_pitch = tnote ;
			te_outp->outevent.midinote_vel = 0 ;
			te_outp->outevent.midinote_onoff = NOTEUP;
			te_outp->outevent.midichan = tchan ;
			insert_event (te_outp) ;
		} 
	}
	fclose(infile) ;
}


/* signal catcher for keyboard termination */
/* PORTNOTE: UNIX */

void sig_int()
{	end_experiment = 1 ;
}

/* stripped-down output-only version of sched (modified from sched.c)
 */
simple_sched()
{	

	
	long finaltime;	
	unsigned long curtime ;

	struct tap_event *locte_p, *tmp_p ;


	curtime = gettodms() ;

	finaltime = curtime - trial_starttime ;

	/* NOTE: should block signals here? */
	while ((locte_p = te_schedlist.te_p) != NULL && 
			locte_p->outevent.time < finaltime) {

		WriteMidiEvent(locte_p) ;

		/* remove the event from queue (necessary!) and  put 
		 * it back on the free list (unnecessary for single
		 * shot playtap  
		 */
		tmp_p = te_schedlist.te_p ;
		te_schedlist.te_p = tmp_p->te_p ;
		free_te(tmp_p) ;
	}
	/* Reset signals here? */
}


/* dummies to resolve references... */

int	mask_chan_param ; 
int	mask_chan_param ; 
int	mask_note_param ; 
int	met_chan_param ; 
int	met_chan_param ; 
int	met_note_param ; 
