/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at finney.17@osu.edu.
 */

/*
 * Although this program is provided "as is", I have taken some pains to
 * try to make it robust and general purpose. I will certainly try to
 * respond to bug reports, bug fixes, suggestions, and comments. Although
 * I don't know of any major bugs, there are certainly some rough spots,
 * and the (verbose) comments in the code (particularly in the headers)
 * are somewhat out of date (but they should be mostly correct). They'll
 * be cleaned up someday. This program has been compiled with gcc.
 */

#define METRON_DEBUG 0
#define CLICK_DEBUG 0


/* put out MIDI active sense every 200 ms; this fixes up a problem I was 
 * having with the TX81Z and masking noise. Setting this to 0 and 
 * recompiling  should probably turn this off, but this hasn't been tried 
 * recently.  
 */
#define ACTIVE_SENSE_ON 1


/* NOTE: keystroke triggers affect the keystroke they occur
 * on. Should metronome triggers do the same thing? Metronome pitch
 * perturbations (yes) vs sync continuation (after is better? but
 * maybe should be done through time.
 */

/*********************************************************************
 * This contains the code for internally-generated MIDI events such as
 * pacing (metronome) tones, masking tones, and also the new code for reading
 * in (fully specified) events from an exernal file (a click file). 
 *
 * The internally generated pacing sounds may be regular or patterned, playing
 * for a fixed number of beats or cycles. All sounded beats have the same 
 * (parameterized) characteristics, although some of these can be changed on
 * the fly by trigger events.  
 *
 * The metronome is always logically running and may generate trigger events 
 * whether it is sounding or not.  The metron_on parameter must be set for 
 * the metronome to sound. metron_on can be turned on and off by a trigger 
 * event (note that metronome trigger events apply to underlying logical 
 * beats, whether sounded or not, and regardless of an imposed pattern. 
 *
 * The "metronome" (or pacing signal, or synchronization signal) is a 
 * likely area of future work. In particular, it would be  nice to allow
 * perturbations of the metronome (e.g., phase switches, tempo changes, 
 * single perturbations (delay on one beat), etc), based on trigger events. 
 * This will require that the metronome be subject to mapping. 
 *
 * NOTE: this code has undergone some major changes over time. Originally,
 * metronome events were prescheduled, and metronome triggers took
 * effect 10 ms after the key release. In the current Linux version (with
 * it's tight central loop), metronome events are scheduled
 * immediately, and trigger events take place at the metronome onset.
 * I'd actually need to check the code again to make sure. However, it
 * looks like the MET_FUDGE #define is defunct, so comments referring to
 * it have been removed.
 *
 * The underlying logical, primitive beat (specified  by MSPB) is the counter 
 * for trigger events.
 *
 * NOTE: this code is permissive in terms of what it lets the
 * user do, e.g., specifying a metronome mode change at a beat within
 * a pattern is not reasonable but is not prevented, and there are some 
 * funny combinations of trigger events.
 *
 * 3/28/00, sf, send MIDI active sense every 200 ms (see ACTIVE_SENSE_ON
 * #defines). This fixes an issue where the Linux drivers send an active
 * sense on device close, which causes masking noise to go off. The active
 * sense messages do not go into the scheduling queue, and do not get
 * written to the output file.
 * 4/9/00, sf, add more metronome parameters so it can be structured
 * by note, vel, len, and MIDI channel. Change existing arrays to be
 * homogenously names. Toyed with the idea of making the related CNT
 * parameters as separate full integer parameters (which could be changed by
 * triggers, allowing for changing the _range_ of the fixed array used),
 * but decided it wasn't useful enough to be worth it. Note, however, that
 * the lengths of the arrays for the different parameters is not required
 * to match. See also params.c.
 * THESE PARAMETERS, IF SET, OVERRIDE THE FIXED METRONOME  PARAMETERS.
 */

/* define this for metronome triggers affecting the metronome beat
 * itself. The old method involved delaying for 10 ms after the metronome
 * beat went _off_, and involved puting the metronome triggers into the
 * output queue. Immediate triggers mean we need to store the trigger
 * data here, using the structures and approach of ftap.c
 */

#define MET_TRIG_IMMED	1

#include <sys/types.h>
#include <sys/time.h>
#include <stdio.h>
#include <errno.h>
#include "ftaplimits.h"
#include "tapmidi.h"
#include "params.h"


/* how many ms to schedule metronome ahead of current time: FIXME - any
 * purpose? In fact, setting it to non-zero screws up ordering of output
 * write (metron triggers are funny). FIXME.
 */
#define METRON_LOOKAHEAD 0



/* times in milliseconds (see above). Issue: longs vs integers vs 
 * unsigned? 
 */
unsigned long last_metron ;
long beat_cnt ;			/* count of beats (sounded or not) sent
				 * out so far.
				 */

#if ACTIVE_SENSE_ON 

#define ACTIVE_SENSE_TIME 200	/* write it every 200 ms */
unsigned int active_sense_time ;

#endif



/* copy stored values to active working variables. Do in global gettimeofday
 * reference
 */
start_metronome ()
{
	last_metron = trial_starttime ;	/* Relative to trial start*/
	beat_cnt = 0 ;

#if ACTIVE_SENSE_ON 
	active_sense_time = trial_starttime + ACTIVE_SENSE_TIME ;
#endif
}

/* schedule any necessary metronome events, given that current time is currtime;
 * if a metronome beat is due, write BOTH the notedown and note up. Assumes that
 * the user/programmer is intelligent enough that things are set up so, e.g.,
 * the metronome off will occur before the next metronome on!
 *
 * Note: perhaps should optimize this for the case when there's nothing to do?
 * But the logical metronome needs to increment in any case.
 *
 * Global last_metron holds the time of the last metronome beat. Once we've
 * established that there _is_ a metronome beat, use next_metron. Given the
 * current small time scale, in no realistic situation should there be more
 * than one beat issued in this loop.
 */



check_metron(currtime)
unsigned long currtime ;
{	
	
	int i ;
	unsigned long endtime ;
	int  debug_cnt = 0 ;	/* sf's debugging variable */
	unsigned long next_metron ;
	endtime = currtime + METRON_LOOKAHEAD ; /* FIXME; see above */

#if METRON_DEBUG
	printf("check_metron: endtime %lu currtime %lu last_metron" 
		" %lu timeofday %lu mspb%d\n",
		endtime, currtime, last_metron, gettodms(), mspb_param) ;
#endif

	while (endtime >= last_metron + mspb_param) {
		debug_cnt ++ ;
		if (debug_cnt ==  20) {	 /* 20 is arbitrary, but only 1 message */
			printf("INTERNAL WARNING: check_metron(), debug_cnt = 20, " 
				"last_metron = %lu\n", last_metron) ;
		}
		last_metron += mspb_param ;
		next_metron = last_metron ;

		beat_cnt++ ;
                /* NOTE: comments here are out of date!!
		 *
 		 * Check if there is a metronome trigger event; if so,
                 * is_metronome_event will remove it from the list. Triggering
                 * occurs from the output scheduling queue, so write a dummy
                 * trigger tap event for scheduling.
                 */

		while  ((i = is_metronome_event (beat_cnt)) >= 0) {

/* an ugly #if, but I don't want to rip out the old metronome trigger
 * code until I know it's unnecessary...code borrowed from ftap.c
 * keystroke events.
 */
#if MET_TRIG_IMMED
			extern struct tap_event midistore[] ;
			extern int		midistore_index ;
			struct tap_event *tsp ;
			ev_exec_action (i) ;
	
			/* Fill in an midistore  entry
			 * (will not go to stdout), making 
			 * special use of available fields.
			 */

			tsp = &(midistore[midistore_index]) ;
			if (++midistore_index > MAXNOTES){
				printf("ERROR: MIDISTORE OVERFLOW!!\n");
				midistore_index-- ;
				continue ;
			}
			tsp->te_type = TE_TRIGGER;
			tsp->te_io = TE_INPUT ;
			tsp->te_sequenceno = 0 ;
			tsp->inevent.midinote_vel = i ;
			tsp->inevent.midinote_pitch = trig_event_table[i].id;
			tsp->inevent.time = next_metron;
			tsp->inevent.midinote_onoff = TRIG_MET_TYPE ;

#else


			/* The 10 ms  in the following is just a fudge factor */
		      	sched_midi_note (0, trig_event_table[i].id, i,
				TRIG_MET_TYPE, TE_TRIGGER, 
				next_metron + met_len_param + 10) ;
#endif

		}

		/* schedule to sound the beat, if necessary (both the
		 * note on and the note off). This now includes the complex
		 * stuff for patterning the metronome with note or
		 * velocity.
		 */
		if (metron_on_param) {
                    int tmp_beat_cnt ;
		    tmp_beat_cnt = beat_cnt - 1 ;
		    if (met_pattern_array_cnt_param == 0 ||
	    	    	    met_pattern_array_list_param[(tmp_beat_cnt) % 
			     met_pattern_array_cnt_param]){
			int vel, len, note, chan ;

			if (met_chan_array_cnt_param)
				chan = met_chan_array_list_param[tmp_beat_cnt %
					met_chan_array_cnt_param] ;
			else
				chan = met_chan_param ;

			if (met_note_array_cnt_param)
				note = met_note_array_list_param[tmp_beat_cnt %
					met_note_array_cnt_param] ;
			else
				note = met_note_param ;

			if (met_vel_array_cnt_param)
				vel = met_vel_array_list_param[tmp_beat_cnt %
					met_vel_array_cnt_param] ;
			else
				vel = met_vel_param ;

			if (met_len_array_cnt_param)
				len = met_len_array_list_param[tmp_beat_cnt %
					met_len_array_cnt_param] ;
			else
				len = met_len_param ;

		        sched_midi_note(chan, note, vel, NOTEDOWN, 
			    TE_NOTE | TE_METRON, next_metron) ;
		        sched_midi_note(chan, note, 0, NOTEUP, 
			    TE_NOTE | TE_METRON, next_metron + len ) ;
			}
		}
	}
#if ACTIVE_SENSE_ON 
	if (active_sense_time <= currtime) {
		write_active_sense() ;
		active_sense_time += ACTIVE_SENSE_TIME ;
	}
#endif

}

/* masking noise always scheduled to start NOW */

mask_on()
{
	if (mask_on_param)
		sched_midi_note(mask_chan_param, mask_note_param, 
			mask_vel_param, NOTEDOWN, TE_NOTE | TE_JUNK, 
			gettodms()) ;
}


mask_off()
{
	if (mask_on_param)
		sched_midi_note(mask_chan_param, mask_note_param, 0, 
			NOTEUP, TE_NOTE | TE_JUNK, gettodms()) ;
}

/****************************************************************************
 * Click file support.
 */

/* function for reading in an external event file (click file) and writing 
 * it to the scheduler. Modified from playtap.c/setup_sched(). NOTE THAT THE
 * INPUT FILE MUST BE FULLY SPECIFIED (e.g., both note downs and note ups);
 * also, the type field will just be copied from the input. Logically, it
 * could be "M" or "F".
 * Issues of combining this with an internally generated metronome have 
 * not been thought through; test it before counting on it!
 * Offset parameter: allow offsetting the input values, so the same
 * layout can be use phase shifted. Possibly should add a multiplier
 * as well?
 * Note that, for now, these values cannot cause trigger events, nor will
 * they be affected by trigger events.
 */


/* size limit for input line */
#define MAX_LINE 100

sched_click(infilename, offset) 
char *infilename ;
int  offset ;
{
	FILE *infile ;
	char buf[MAX_LINE + 1] ;
	struct tap_event *te_outp ;	
	int i ;
	
	/* fields for data lines in input file... */
	int time, midichan, notenum, velocity, sequence ;

	/* do single chars as strings, since scanf doesn't do field separation 
	 * for characters.
	 */ 
	char 	notename[10], updown[10], typestr[10] ;

	unsigned char updownchar ;

	char type ;

	infile = fopen (infilename, "r") ;
	if (infile == NULL) {
		printf("sched_click error: cannot open input file %s\n", 
			infilename) ;
		exit (-1) ;
	}
	/* note: fgets will include the "\n" in buf (unlike gets() ) */
	while (fgets (buf, MAX_LINE, infile) != NULL) {
		if (strlen (buf) == 0 || buf[0] == ' ' || buf[0] == '\t' ||
			buf[0] == '\n' || buf[0] == '#')
			continue ;

		i = sscanf (buf, "%d%s%d%d%s%d%d%s", &time, updown, &midichan,
			&notenum, notename, &velocity, &sequence, typestr ) ;
		type = typestr[0] ;
		if (i != 8) {
                	buf[strlen(buf) - 1] = '\0' ;
			printf("sched_click: sscanf error  %d, file %s, line %s\n", 
				i, infilename, buf) ;
			exit(-1) ;
		}
		te_outp = get_te() ;
		te_outp->outevent.time = trial_starttime + (int) time + offset;
		te_outp->outevent.midichan = midichan ;
		te_outp->outevent.midinote_pitch = notenum ;
		te_outp->outevent.midinote_vel = velocity ;
		if (updown[0] == NOTEDOWN_CHAR)
			updownchar = NOTEDOWN ;
		else if (updown[0] == NOTEUP_CHAR)
			updownchar = NOTEUP ;
		else {
			printf("Bad event in clickfile %s, updown = %c\n",
				infilename, updown[0]) ;
			free_te (te_outp) ;
			continue;
		}
		te_outp->outevent.midinote_onoff = updownchar;
	
		/* FIXME: an unhappy consequence of the factoring out
		 * of te_type.
		 */
		te_outp->te_type = TE_NOTE ;
		te_outp->te_io =  TE_OUTPUT ;
		if (typestr[0] == 'F')
			te_outp->te_type |= TE_KEYSTROKE ;
		else if (typestr[0] == 'M')
			te_outp->te_type |= TE_METRON ;
		else {
			printf("Bad event in clickfile %s, type = %c\n",
				infilename, typestr[0]) ;
			free_te (te_outp) ;
			continue;
		}
				
#if CLICK_DEBUG
	printf("sched_click(): about to insert tep\n") ;
	PrintTapEvent (te_outp) ;
#endif
		insert_event (te_outp) ;
	}
	fclose(infile) ;
}
