/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at finney.17@osu.edu.
 */

/*
 * Although this program is provided "as is", I have taken some pains to
 * try to make it robust and general purpose. I will certainly try to
 * respond to bug reports, bug fixes, suggestions, and comments. Although
 * I don't know of any major bugs, there are certainly some rough spots,
 * and the (verbose) comments in the code (particularly in the headers)
 * are somewhat out of date (but they should be mostly correct). They'll
 * be cleaned up someday. This program has been compiled with gcc.
 */

/* OS (Linux) specific code. Many (though not all) of the Linux dependencies
 * are coded here. Some are scattered throughout the code (sometimes with
 * a "PORTNOTE" comment), and MIDI-specific code is in  linux_midi.c 
 * If a port to another OS is attempted, all Linux-specific code should be
 * moved to this module, and a set of calls which generalizes across OS's
 * should be designed.
 *
 * Anyone calling functions returning values here should include linux.h:
 * 
 * To do: Add more diagnostics for RTC efficiency? 
 * 
 * NOTE: errors here will usually abort/exit, though sometimes they'll print
 * out a message and continue.
 *
 * EXPORTED FUNCTIONS:
 *	InitRealTime()
 *	RTCNap()
 *	StartRTC()
 *	StopRTC()
 */



#include <linux/mc146818rtc.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <limits.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <sched.h>
#include <sys/mman.h>

#include "linux.h"



int running_as_root = 0 ;


/***************************************************************************
 *  "Real time" and timing  stuff 
 */


/* Set up to make this process run fast...
 * Linux  FIFO priorities are 1-99, higher is better.
 *
 */

#define RT_PRI 60

void
InitRealTime()
{
	struct sched_param sp ;

	/* structure to grab a lot of (locked) stack space for future use
 	 * (see mlockall() man page)
	 */
	int dummy[10000] ;

	dummy[0] = 1 ;
	dummy[9999] = 1 ;

/* following calls will fail if not running with root privileges, but
 * things should run OK anyway..probably.
 */


	if (mlockall(MCL_CURRENT | MCL_FUTURE) < 0) {
/*
		perror ("Warning: mlockall") ;
 */
	};
	sp.sched_priority = RT_PRI ;
	if (sched_setscheduler (0, SCHED_FIFO, 	&sp) < 0) {
 /*
		perror ("Warning: sched_setscheduler") ;
 */
	};
}


/* routine to sleep for .49 (1000/2048) milliseconds. Must be run by root,
 * and can only be run by one user at a time. Uses periodic interrrupt (PIE)
 * mode of /dev/rtc.
 * 
 * timedata & 0xFFF0 >> 8 should contain number of missed interrupts if it
 * needs to be tracked here.
 * 
 */




int rtfd ;


void 
RTCNap()
{

	unsigned long timedata ;
	int retval ;

	/* FIXME: if not running as root, we won't be hogging the
	 * processor, so don't bother killing time.
 	 */
	if (!running_as_root)
		return ;

	if ((retval = read (rtfd, &timedata, sizeof (unsigned long))) < 0) {
		perror ("/dev/rtc: read") ;
		exit (-1) ;
	}
}

/* allow to run as non-SU for debugging purposes (values here: 2048 and 64). 
 * (???)
 */
		
#define INTR_HZ 2048
#define NOT_SU_INTR_HZ 64

void 
StartRTC()
{
	int retval, intr_hz ;
	if ((rtfd = open ("/dev/rtc", O_RDONLY)) < 0) {
		perror ("/dev/rtc: open") ;
		exit (-1) ;
	}
	if (geteuid() == 0) {
		intr_hz = INTR_HZ ;
		printf("Running with superuser privileges\n") ;
		running_as_root = 1 ;
	}
	else {
		running_as_root = 0 ;
		printf("Running as normal user\n") ;
		intr_hz = NOT_SU_INTR_HZ ;
	}
	if ((retval = ioctl (rtfd, RTC_IRQP_SET, intr_hz)) < 0) {
		perror ("/dev/rtc: IRQP_SET") ;
		exit (-1) ;
	}
	if ((retval = ioctl (rtfd, RTC_PIE_ON, 0)) < 0) {
		perror ("/dev/rtc: PIE_ON") ;
		exit (-1) ;
	}
}


void 
StopRTC()
{
	int retval ;
	if ((retval = ioctl (rtfd, RTC_PIE_OFF, 0)) < 0) {
		perror ("/dev/rtc: PIE_OFF") ;
		exit (-1) ;
	}
	close (rtfd) ;
}

/* Get current time of day in milliseconds; this will be  ms since 
 * 1/1/1970. 
 */

/* could probably make this a define rather than a function... */

unsigned long tv_to_ms(tvp)
struct timeval *tvp ;
{
	return (tvp->tv_sec * 1000  + tvp->tv_usec/1000) ;
}

unsigned long 
gettodms()
{	struct timeval tv ;
	gettimeofday (&tv, NULL) ;
	return (tv_to_ms (&tv)) ;
}


/********************************************************************************
 * Miscellaneous...
 */

init_rand()
{
	srand (getpid()) ;
}

/* get an integer random number between 'start' and 'end' (changed from
 * earlier intended start + interval).
 */

int getrand(start, end) 
int start, end ;
{	int i ;
	i = rand() % (end - start + 1) ;
	return (start + i ) ;
}

	
