/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at finney.17@osu.edu.
 */

/*
 * Although this program is provided "as is", I have taken some pains to
 * try to make it robust and general purpose. I will certainly try to
 * respond to bug reports, bug fixes, suggestions, and comments. Although
 * I don't know of any major bugs, there are certainly some rough spots,
 * and the (verbose) comments in the code (particularly in the headers)
 * are somewhat out of date (but they should be mostly correct). They'll
 * be cleaned up someday. This program has been compiled with gcc.
 */


/* The MIDI-related routines, both input and output.  Raw MIDI byte streams
 * ("/dev/midi") are used.  The ReadMidiEvent and WriteMidiEvent routines take 
 * the internal tap_event structure and transfer the fields from/to the 
 * appropriate form for actual I/O. ReadMidiEvent allocates the necessary
 * tap_event structure when there's something to return.
 * Input time stamping occurs here. Not all possible MIDI events are handled,
 * but everything useful for psych experiments should probably be here.
 *
 * Some  of this is reinventing the wheel (MIDI message parsing); it might
 * (or might not) have been better to find a set of existing routines.
 *
 * NOTES:
 *
 *   Input: NON_BLOCK reads would be preferable to select(), but the 
 *   4front driver doesn't support them.
 *
 *   Output: at the actual driver level, many Linux MIDI drivers only have a
 *   10 ms granularity; this can be tested with the ftaploop() test program.
 *
 *   Other: the MPU-401 MIDI driver seems to send ACTIVE_SENSE, which can
 *   screw up the masking noise on the TX81Z. The MPU-401 also defaults to midi
 *   thru when the device is closed.
 *
 *   In view of this, the use of a simple serial MIDI port should probably be 
 *   considered.
 *
 *   This is messier than it needs to be, but FTAP was originally written for
 *   an SGI which provided time stamped, parsed MIDI message. Existing code
 *   probably does what this program code does. Complexities arise from the
 *   use of a circular buffer, handling of running status,the
 *   fact that an entire MIDI message may not be availalbe when the read is
 *   done, and the need to do ms-precision timestamping. 
 *
 *   Assumption: (locally) polled MIDI reads will occur at least once/ms.
 *   This can be verified in the output file by value of the AV_SCHED
 *   variables.
 *
 *   The MIDI channel in the tap_event is + 1 (the way the MIDI doc
 *   refers to it, channel 1 is coded in the message as 0).
 *
 *   Velocity for NOTEOFF messages (which is non-zero for some keyboards)
 *   is thrown out and set to zero.
 *
 *   In the face of some rare hardware errors, this is becoming a little more
 *   robust to bogus input, HOWEVER, it basically prints a diagnostic and
 *   then continues on anyway. In particular, the handling of running
 *   status on an error is not done in any reasonable fashion.
 *
 *   Perhaps all MIDI I/O should be in terms of status/byte1/byte2,
 *   with byte1 and byte2 aliased to note/vel: updown as additional?
 *   Internally, the mnote struct in a tapevent only handles Noteon/Noteoff info.
 *   The code here uses both a more general MidiEvent structure, and a raw byte
 *   MIDI data stream. Could probably be simplified.
 *
 *
 * TO DO:
 *  1) Should split into linux-specific and more generic portions; also,
 *	input and output.
 *  2) me vs mep is a mess.
 *  3) Be able to send MIDI program change (based on a trigger)?
 *  4)	streamline layers  (get rid of MidiEvent; could occur with
 * 	generalization of tap_event to generic MIDI format.
 *  5) 	clean up MIDI notes off
 *  6) Be able to TOSS OUT MIDI messages which are not explicitly handled,
 *	rather than croaking. This is getting better, but variable length
 *	SYSEX messages will probably never be handled. Some of the stuff
 *	which this has been coded to handle (Channel aftertouch, 
 *	active sense) has not been tested.
 *
 *
 * Externally available functions:
 *	MidiInit
 *	ReadMidiEvent
 *	WriteMidiEvent
 *	SaveBufInit
 *	MidiNotesOff
 *
 *
 * Modification History:
 *   3/13/00, sf, Major hack:  Separate te_type (KEYSTROKE/CONTROLLER/TRIGGER) 
 *	from INPUT/FEEDBACK (latter relevant only for output).
 *   4/3/00, sf, add input time discrepancy diagnostics: maximum time between
 * 	timestamps of an individual  MIDI message.
 *   10/7/00, sf, MidiParse now parses (and discards) program change and
 *	channel aftertouch.
 *   10/17/00, sf, print short diagnostic to screen on bogus NOTEON/NOTEOFF 
 * 	messages (note value =0, > 127, velocity > 127), and write a fuller
 *	message to the output file (perhaps this should appended to a separate 
 *	file; simply change write_log() ).  No clever parse recovery is
 *	done (e.g., looking for next real MIDI status byte); just blunder
 *	on.  Also log and discard (but don't exit) unrecognized status bytes. 
 *	Running status is probably not handled reasonably. Possibly one should 
 *	do some cleanup parsing, but this should _not_ be occuring much (or 
 *	at all), and I'm not taking the time to do it right. Also, quietly 
 *	pass on ACTIVE SENSE (not tested).
 *   2/5/01, sf, quietly ignore MIDI timing clock messages (e.g., as put
 *	out by the Yamaha PortaSound).  Do this in the low-level MidiRawRead
 *	routine, and just toss out _all_ system real-time messages (inludes
 *	ACTIVE SENSING.
 *
 */

#define DEBUG 0



#include <sys/time.h>
#include <sys/types.h>
#include <sys/unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdio.h>
#include <limits.h>
#include <string.h>
#include <signal.h>
#include <errno.h>

#include "tapmidi.h"
#include "linux.h"
#include "params.h"


/* global MIDI fd shared between input and output (Linux seems to only allow
 * a single open on /dev/midi) 
 */

int midifd ;

/* The name of the Linux MIDI device. Should probably be placed somewhere
 * more accessible.
 */

#define MIDI_DEV	"/dev/midi"


/* MIDI status bytes: the only ones we're prepared to encounter. 
 * Note: as long as one is going to handle the (necessary) Control Change
 * messages for piano pedalling, it's easy enough to handle all three-byte 
 * messages (they'll just be passed through).  Channel Pressure aftertouch
 * (as on a DX-7 or DH-100) is a two byte message which requires special
 * case handling; it is simply discarded along with  program  change messages.
 */

#define NOTEOFF_STAT		0x80
#define NOTEON_STAT		0x90

#define POLY_KEYPRESS_STAT	0xA0		/* After touch */
#define CONTROL_CHANGE_STAT	0xB0		/* e.g., Sustain Pedal */
#define PITCH_WHEEL_STAT	0xE0	

#define CHAN_PRESS_STAT		0xD0		/* only 1 data byte */
#define PROG_CHANGE_STAT	0XC0

/* MIDI System Real-Time Message for active sensing, so mask notes don't 
 * get turned off. FTAP sends this out by default every 200 ms 
 * (ACTIVE_SENSE_TIME). 
 */

#define ACTIVE_SENSE_MSG	0xFE

/* SYS_RT_BITS: System Real-Time Message. Double duty as value and bit flag.
 */

#define SYS_RT_BITS		0xF8


/* the raw input (circular) buffer; each byte is time-stamped for now,
 * and the first (data) byte of a message provides the MIDI msg timestamp.
 */
 
#define MAX_INBYTES 200
struct chartime {
	unsigned char c ;
	unsigned long time ;
};

int rawchars_cnt = 0 ;	/* count of available chartime bytes */


/* Internal structure for MIDI messages. This should be fairly generic,
 * excluding long SYSEX messages.
 */

#define MAX_MIDI_LEN	5

struct MidiEvent {
	unsigned char status ;
	int datalen ;
	unsigned char data[MAX_MIDI_LEN] ;	/* doesn't include stat */
	unsigned long time ;
} ;

int MidiParse() ;

struct chartime * cbuf_getchar() ;
void cbuf_ungetchar() ;

/*
 * Defines and routines for saving the last 10 key down events sent out; this
 * is for trying to turn notes off at the end of a trial. At one point
 * early on I tried the MIDI "all notes off" message but couldn't get it
 * to work right. 
 *
 * FIXME: There  must be a better way to do this! Try MIDI all notes off 
 * message again?  Send individual notes off for all notes and channels?
 */

#define SAVEBUF_SZ      10              /* how many */
int     savebuf[SAVEBUF_SZ] ;           /* the storage */
int     chanbuf[SAVEBUF_SZ] ;           /* also need midi channel */
int     savebuf_index ;

/* diagnostic storage; inited to 0 in output.c */
int sched_input_max, sched_input_max_time ;

/* (MIDI) error logging. If there are more than 10 errors in a trial,
 * you're probably in big trouble
 */
#define MAX_LOG_ERRORS 		10
int 	log_errors =		0 ;

struct log_error {
	unsigned long time ;
	int status, note, vel ;
}	log_error [MAX_LOG_ERRORS] ;


/******************************************************************************
 * SETUP CODE
 */


/* NOTE: non-blocking I/O would be preferred for the single process design; 
 * since it doesn't work with the 4front driver, use select().
 */

InitMidi()
{
	if ((midifd = open (MIDI_DEV, O_RDWR | O_NONBLOCK)) < 0) {
		perror (MIDI_DEV) ;
		exit(-1) ;	
	}; 	 
}

/* for good measure, close and reopen between interactive runs */
CloseMidi()
{
	close (midifd) ;
}



/*****************************************************************************
 * INPUT CODE
 */


/* 
 * ReadMidiEvent () allocates and returns a pointer to a filled  
 * tap_event if there's a complete MIDI message available, else NULL.
 * Allocation of the tapevent is done here.
 */


struct tap_event *
ReadMidiEvent()
{
	unsigned int status, status_nibble, velocity, note ;


	struct MidiEvent Mevent ;
	
	struct tap_event *tep ;



#if DEBUG
	printf("ReadMidiEvent: top\n") ;
#endif

	/* accumulate any raw MIDI bytes */
	MidiRawRead() ;

retry:	if (MidiParse (&Mevent) == 0) {
#if DEBUG
		printf("ReadMidiEvent: returning NULL\n") ;
#endif
		return (NULL) ;
	}

	status = Mevent.status ;
	status_nibble = status & 0xF0 ;

	/* fill in the input fields of the tep structure... */

	tep = get_te() ;
	/* input time stamp in msec... */
	tep->inevent.time = Mevent.time;
	tep->inevent.midichan = (status & 0x0F) + 1 ;
	tep->te_io = TE_INPUT ;

	
	/* Do the setup for a Controller event */
	if (status_nibble != NOTEON_STAT && status_nibble != NOTEOFF_STAT) {
		tep->inevent.status = status;
		tep->inevent.byte1 = Mevent.data[0];
		tep->inevent.byte2 = Mevent.data[1];
		tep->te_type |= TE_CONTROLLER ;

	}
	else { /* Otherwise, a note event */

		note = Mevent.data[0];
		velocity = Mevent.data[1];
	
		/* two ways to signal NOTEOFF (ftap will always set the
 		 * velocity to zero) */
		if ((status_nibble == NOTEOFF_STAT) || 
		   (status_nibble == NOTEON_STAT && velocity == 0)) {
			tep->inevent.midinote_onoff = NOTEUP ;
			tep->inevent.midinote_vel =  0 ;
		}
		else {
			tep->inevent.midinote_onoff = NOTEDOWN ;
			tep->inevent.midinote_vel =  velocity;
		}
	
		tep->inevent.midinote_pitch = note;
		if (note == 0  || note > 127 || velocity > 127) {
			printf("MIDI WARNING: check data file\n") ; 
			enter_log (tep->inevent.time -  trial_starttime,
				status & 0xFF, note, velocity) ;
		}
		tep->te_type |= (TE_NOTE | TE_KEYSTROKE) ;
	}
	
#if DEBUG
	printf("ReadMidiEvent: returning tapevent: \n") ;
	PrintTapEvent (tep) ;
	
#endif
	return (tep) ;
		
}

/* Try to parse a (single) MIDI message from the raw queue, and put it into the
 * argument structure. Returns 1 if there is one, 0 if not (insufficient data).
 * Assumes that the read position in rawinbuf is start of a MIDI message.
 *
 * This ends up being pretty messy. (1) A circular buffer is used, although
 * the pointers are handled at a lower level, (2) MIDI running status 
 * complicates message parsing, (3) At the time when this routine is called,
 * a full MIDI message may not be available, (4) Inclusion of MIDI 
 * channel aftertouch means that not all messages are 3 bytes, (5)
 * input messages must be time stamped, (6) it is necessary to return if
 * a full message isn't available; blocking isn't acceptable. If I knew 
 * where to steal (comprehensible)  code for this, I'd do it...there's also
 * some ugly (and probably unnecessary) code for retaining timing stats on
 * the time between input bytes.
 *
 * ALTERNATIVE APPROACH: maintain a partially parsed message here rather
 * than putting bytes back?
 * 
 * Assumption: only one ctp will be available at a time.
 *
 * 10/7/00, add processing of 2-byte channel aftertouch and program
 * 	change messages...for now, discard, and print diagnostics. Makes
 *	this routine even messier, with a goto added (since this routine
 *	is set up to return either with a parsed message or when there's
 *	insufficient data.)
 */

int MidiParse(mep)
struct MidiEvent *mep ;
{	
	struct chartime *ctp ;

	unsigned int time1, time2 ;
	int diff ;

	int byte_read  ;	 /* set if we read a byte looking for status,
				  * but it turns out to be a data byte due
				  * to running status. 
				  */

	unsigned char status ;
	/* save last status byte for possible running status, must save across
	 * calls */
	static unsigned char laststatus = 0;


#if DEBUG
	printf("MidiParse: top\n") ;
#endif




retry:	byte_read = 0 ;
	if (rawchars_cnt) {	/* nothing to do unless there's _some_ data.. */
		ctp = cbuf_getchar () ;
		status = ctp->c ;
		time1 = ctp->time ;
		if ((status & 0x80) == 0) {	/* running status */
			status = laststatus ;
			byte_read = 1 ;
		}
		else {
			byte_read = 0 ;
			laststatus = status ; /* doesn't hurt to do it now */
		}
		switch (status & 0xF0)
		{	


			/* all have 2 data bytes */
			case NOTEON_STAT:
			case NOTEOFF_STAT:
			case POLY_KEYPRESS_STAT:
			case CONTROL_CHANGE_STAT:
			case PITCH_WHEEL_STAT:
				/* if not enough data for a full message .. */
				if (rawchars_cnt + byte_read < 2) {
					if (byte_read)
						cbuf_ungetchar() ;
					return (0) ; /* not a full message */
				}
				mep->status = status ;
				mep->datalen = 2 ;
				/* if byte_read is set, ctp is already
				 * set to the first data byte.
				 */
				if (!byte_read) 	
					ctp = cbuf_getchar () ;
				mep->data[0] = ctp->c ;

				/* take the time from the first data byte 
				 * (so we don't have to special case running 
				 * status).
 	 			 */
				mep->time = ctp->time ;
				ctp = cbuf_getchar () ;
				time2 = ctp->time ;
				if ((diff = (time2-time1)) > sched_input_max) {
					sched_input_max = diff ;
					sched_input_max_time = gettodms() - 
						trial_starttime ;
				}

				mep->data[1] = ctp->c ;
	
#if DEBUG
    				printf("MidiParse: Parsed a full message\n") ;
				PrintMidiEvent(mep) ;
#endif
				return (1) ;

				break ;

			/* Messages with a single data byte;both are discarded.
			 * Copying a bunch of code from above, though running
			 * stat may be overkill
			 */

			case CHAN_PRESS_STAT:
			case PROG_CHANGE_STAT:
				/* if not enough data for a full message .. */
				if (rawchars_cnt + byte_read < 1) {
					if (byte_read)
						cbuf_ungetchar() ;
					return (0) ; /* not a full message */
				}
				/* get and toss the data byte */
				if (!byte_read) 	
					ctp = cbuf_getchar () ;
	
/*     				
	printf("WARNING: discarding a 0x%x input MIDI message\n", status) ; 
 */
				goto retry ;

			

			default: 
				printf("MIDI WARNING: check data file\n") ; 
				enter_log (time1 - trial_starttime,
					status & 0xFF, 0, 0) ;
				goto retry ;
		}
	}
	return (0) ;
}
	


/* MidiRawRead: read up to MAXREAD raw bytes  from the MIDI interface ;
 * this will usually only be 1 byte if running as a a real-time process.
 * 
 * Would prefer to use O_NONBLOCK, but select() will do.
 *
 * Still messy:
 * Assumed: (1) cbuf_getchar() will be called only when rawchars_cnt > 0.
 * Now: toss out any system RT bytes (could also do SYS COMM, but this
 * way, you'll get better diagnostics with SYSEX).
 */


/* the following circular buffer globals are "hidden" down here...
 */

struct chartime  rawinbuf[MAX_INBYTES] ;

int c_readpos = 0 , c_writepos = 0 ;

# define MAXREAD 15
MidiRawRead()
{
	unsigned char tmpbuf[MAXREAD] ;
	unsigned long curtime ;
	char errbuf[100] ;
	int i, n, retval ;
	
	struct timeval tv = {0, 0} ;

        fd_set fds ;

        FD_ZERO (&fds) ;
        FD_SET (midifd, &fds) ;

#if DEBUG
	printf("MidiRawRead entry: rawchars_cnt %d, c_readpos %d, " 
		"c_writepos %d\n", rawchars_cnt, c_readpos, c_writepos) ;
#endif


retry:  retval = select (midifd + 1, &fds, NULL, NULL, &tv ) ;
	if (retval == 0)
		return ;
        if (retval < 0) {
		if (errno == EINTR)
			goto retry; 	/* when trying to do profiling of 
					 * this code, this error seems to 
					 * occur.
					 */
                perror ("MidiRawRead: select() ") ;
		exit (-1) ;

        }


	

	n = read (midifd, tmpbuf, MAXREAD) ;
	/* Check for non-blocking read error */
	if (n <= 0 && errno == EAGAIN)
		return ;	/* no data, but not an error */
	if (n <= 0) {
		sprintf(errbuf, "MidiRawRead returned %d", n) ;
		perror (errbuf) ;
		exit (-1) ;
	}
	curtime = gettodms() ;
#if DEBUG
    	printf ("MidiRawRead: got %d bytes\n", n) ;
#endif
	for (i = 0 ; i < n; i++) {
		/* NOTE: toss MIDI System Real-Time Messages */
		if ((tmpbuf[i] & SYS_RT_BITS) == SYS_RT_BITS)
			continue ;
		rawinbuf[c_writepos].c = tmpbuf[i] ;
		rawinbuf[c_writepos].time = curtime ; 
#if DEBUG
    	printf ("c = 0x%x, time = %lu\n", tmpbuf[i],   curtime) ;
#endif
		c_writepos++ ;
		rawchars_cnt++ ;
		if (c_writepos == MAX_INBYTES) {
#if DEBUG
    			printf ("Resetting c_writepos\n") ;
#endif
			c_writepos = 0 ;
			/* error should only occur with > 200 bytes/scheduling
			 * unit (only 1 ms if running setuid root, 16 ms if 
		 	 * testing as normal user). */
			if (c_writepos == c_readpos) {
				printf("MidiRawRead: Buffer Overflow!!\n") ;
				exit (-1) ;
			}
		};	
	
	}
#if DEBUG
	printf("MidiRawRead exit: rawchars_cnt %d, c_readpos %d, " 
		"c_writepos %d\n", rawchars_cnt, c_readpos, c_writepos) ;
#endif
}

/* utility procedures so code above doesn't have to worry about 
 * circular buffer.
 */
struct chartime *cbuf_getchar()
{	
	struct chartime *ctp ;

#if DEBUG
	printf("cbuf_getchar entry: rawchars_cnt %d, c_readpos %d, " 
		"c_writepos %d\n", rawchars_cnt, c_readpos, c_writepos) ;
#endif

	if (rawchars_cnt <= 0) {
		printf("cbuf_getchar: no characters!!\n") ;
		exit (-1) ;
	}
	ctp = & (rawinbuf[c_readpos]) ;
	c_readpos++ ;
	rawchars_cnt -- ;
		

	if (c_readpos == MAX_INBYTES)
		c_readpos = 0 ; 

#if DEBUG
	printf("cbuf_getchar: returning 0x%x\n", (ctp)->c);
#endif

	return (ctp) ;
}

/* put a character back (just backup pointers) */

void 
cbuf_ungetchar()

{	

	rawchars_cnt++ ;

	c_readpos-- ;

	if (c_readpos < 0) 
		c_readpos = MAX_INBYTES - 1 ;
#if DEBUG
	printf("cbuf_ungetchar exit: rawchars_cnt %d, c_readpos %d, " 
		"c_writepos %d\n", rawchars_cnt, c_readpos, c_writepos) ;
#endif
}


/*****************************************************************************
 * OUTPUT CODE
 */
		

/* For simplicity, output doesn't use running status. Output is done
 * immediately; no time stamps. Direct translation from mnote to output
 * buffer (no intervening MidiEvent).
 */

/* FIXME: assuming 3 byte messages */

#define MAX_MIDI_OUTPUT_LEN 10


WriteMidiEvent(tep)
struct tap_event *tep ;
{	
	unsigned char outbuf[MAX_MIDI_OUTPUT_LEN];   /* overflow not checked */
	int onoff, retval ;

#if DEBUG
	printf("WriteMidiEvent, writing:") ;
	PrintTapEvent (tep) ;
#endif

	if (tep->te_type  & TE_CONTROLLER) {
		outbuf[0] = tep->outevent.status ;
		outbuf[1] = tep->outevent.byte1 ;
		outbuf[2] = tep->outevent.byte2 ;
	}
	else {

		if (tep->outevent.midinote_onoff == NOTEDOWN)
			onoff = NOTEON_STAT ;
		else
			onoff = NOTEOFF_STAT ;
	
	
		
		outbuf[0]= onoff | tep->outevent.midichan - 1 ;
		outbuf[1] = tep->outevent.midinote_pitch;
		outbuf[2] = tep->outevent.midinote_vel;
	
		/* save it if it's a NOTEDOWN event */
		if (onoff == NOTEON_STAT) {
			savebuf[savebuf_index] = 
				tep->outevent.midinote_pitch ;
			chanbuf[savebuf_index] = 
				tep->outevent.midichan  ;
			savebuf_index = 
				(savebuf_index + 1) % SAVEBUF_SZ;  /* circular*/
		}
	
	}
#if DEBUG
	printf ("MIDI output event: 0x%x 0x%x 0x%x, schedtime %lu curtime %lu\n",
		outbuf[0], outbuf[1], outbuf[2], tep->outevent.time, 
		gettodms() ) ;
#endif
	/* We should  be prepared
 	 * to handle user (ctl-C) interrupts. Anything else is probably
	 * a real error; either caller should check, or we should bail
	 * here
	 */
	if (write (midifd, outbuf, 3) < 3) {
		perror ("MIDI write failure") ;
		exit (-1) ;		/* Add more error checking? */
	}
}


unsigned char active_sense_buf = ACTIVE_SENSE_MSG ;

/* send an active sensing message; called from check_metron(). This
 * does not go through the output queue  
 */
void
write_active_sense()
{
	if (write (midifd, &active_sense_buf, 1) != 1) {
		perror ("MIDI write failure") ;
		exit (-1) ;		/* FIXME: error checking */
	}
}

/*********************************************************************
 * CLEANUP CODE
 */

/* Unfortunately, neither mdPanic nor actually sending a notes off message
 * seems to work right (at least, in terms of actually stopping  a sounding
 * MIDI note on my setup of 1999 or so), so do stuff by hand.
 * Gratuitous note offs shouldn't hurt.
 */



InitSaveBuf()
{       int i ;
        for (i = 0 ; i < SAVEBUF_SZ; i ++)
                savebuf[i] = 0 ;
        savebuf_index = 0 ;
}

/* This routine writes NoteOffs for the last 10 notes sent; it's somewhat
 * messy, but previous use of simpler mechanisms hasn't worked.  Perhaps
 * time to try MIDI all notes off (per channel) message again: Bn 123 0.
 * Another alternative would be to send all possible notes off on all 
 * channels. 
 */

MidiNotesOff()
{
	int i, j ;
	unsigned char outbuf[MAX_MIDI_OUTPUT_LEN];  /* overflow not checked */

	/* explicitly turn off masking noise */
	
	if (mask_chan_param >= 1) { /* if 0, midi interface will complain */
		outbuf[0] = NOTEOFF_STAT | mask_chan_param - 1 ;
		outbuf[1] = mask_note_param ;
		outbuf[2] = 0;
		if (write (midifd, outbuf, 3) < 3) {
			perror ("MIDI write failure") ;
			exit (-1) ;		/* FIXME: error checking */
		}
	}

	/* explicitly turn off possible metronome  */

	if (met_chan_param >= 1) {
		outbuf[0]  = NOTEOFF_STAT | met_chan_param - 1 ;
		outbuf[1] = met_note_param ;
		outbuf[2] = 0;
		if (write (midifd, outbuf, 3) < 3) {
			perror ("MIDI write failure") ;
			exit (-1) ;		/* FIXME: error checking */
		}
	}

	/* Turn off last N notes sent:  assume a 0 note is an 
 	 * initialization 
	 */
	for (i = 0 ; i < SAVEBUF_SZ ; i++) {
		if (savebuf[i] == 0)
			continue; 
		outbuf[0] = NOTEOFF_STAT | chanbuf[i]- 1 ;
		outbuf[1] = savebuf[i];
		outbuf[2] = 0;
		if (write (midifd, outbuf, 3) < 3) {
			perror ("MIDI write failure") ;
			exit (-1) ;		/* FIXME: error checking */
		}
		
	}



}


/*********************************************************************
 * DIAGNOSTIC CODE
 */


/*  add MIDI channel printout? */

PrintTapEvent(tep)
struct tap_event  *tep ;
{
	int type ;
	type = tep->te_type ;
        printf("  tap_event()\n") ;
        printf("  type: 0x%x\n",  type) ;
	if (type & TE_CONTROLLER)  {
	        printf("  Inevent CONT: time %lu chan %d " 
				"status 0x%x byte1 0x%x byte2 0x%x\n",
	                tep->inevent.time, tep->inevent.midichan,
			tep->inevent.status & 0xFF,
			tep->inevent.byte1 & 0xFF,
			tep->inevent.byte2 & 0xFF);
	        printf("  Outevent CONT: time %lu chan %d "
				"status 0x%x byte1 0x%x byte2 0x%x\n",
	                tep->outevent.time, tep->outevent.midichan,
			tep->outevent.status & 0xFF,
			tep->outevent.byte1 & 0xFF,
			tep->outevent.byte2 & 0xFF);
		return ;

	}
	/* otherwise, default note printout */
        printf("  Inevent NOTE: time %lu chan %d note 0x%x vel %d onoff 0x%x\n",
                tep->inevent.time, tep->inevent.midichan,
		tep->inevent.midinote_pitch, 
		tep->inevent.midinote_vel,
                tep->inevent.midinote_onoff) ;
        printf("  Outevent NOTE: time %lu chan %d note 0x%x vel %d onoff 0x%x\n",
                tep->outevent.time, tep->outevent.midichan,
		tep->outevent.midinote_pitch, 
		tep->outevent.midinote_vel,
                tep->outevent.midinote_onoff) ;

}



PrintMidiEvent(mep)
struct MidiEvent *mep ;
{	int i ;
	printf ("  MidiEvent: status 0x%x, datalen %d, time %lu\n", 
		mep->status & 0xFF, mep->datalen, mep->time) ;
	printf ("  data: ") ;
	for (i = 0 ; i < mep->datalen ; i++) 
		printf("0x%x ", mep->data[i] & 0xFF) ;
	printf("\n") ;
}


/* save bogus MIDI data to be written out later. Only 10 entries will
 * be saved; you should never need this many! note and vel will be zero
 * if it's an unrecognized status byte.
 * NOTE: unfortunately, this data has already been partially processed, e.g.,
 * status bytes from running status have been interpolated. It would be
 * nice (for diagnostic purposes) to have access to the actual raw data
 * stream here, but the code currently makes that difficult.
 */

enter_log (time, status, note, vel)
unsigned long time ;
int status, note, vel ;
{
	if (log_errors < MAX_LOG_ERRORS) {
		log_error[log_errors].time = time ;
		log_error[log_errors].status = status ;
		log_error[log_errors].note = note ;
		log_error[log_errors].vel = vel ;
	}
	log_errors += 1 ;
		
}

write_log (outfile)
FILE *outfile ;
{	int i, cnt  ;
	if (log_errors > MAX_LOG_ERRORS)
		cnt = MAX_LOG_ERRORS ;
	else
		cnt = log_errors ;
	if (cnt == 0)
		return ;
	fprintf(outfile, "#\n") ;
	for (i = 0 ; i < cnt; i ++) {
		fprintf(outfile, "# MIDI_ERROR\t%lu\t0x%x\t%d\t%d\n",
			log_error[i].time, log_error[i].status,
			log_error[i].note, log_error[i].vel) ;
	}
	fprintf(outfile, "#\n") ;
}
