/************************************************************************
 * ftap: a Linux-based, MIDI-based program for running tapping and music
 * experiments.
 * 
 * Copyright (C) 1999, 2000 Steven A. Finney
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Steve Finney can be reached by email at finney.17@osu.edu.
 */

/*
 * Although this program is provided "as is", I have taken some pains to
 * try to make it robust and general purpose. I will certainly try to
 * respond to bug reports, bug fixes, suggestions, and comments. Although
 * I don't know of any major bugs, there are certainly some rough spots,
 * and the (verbose) comments in the code (particularly in the headers)
 * are somewhat out of date (but they should be mostly correct). They'll
 * be cleaned up someday. This program has been compiled with gcc.
 */



/* 
 * This is the big mondo main file for the FTAP program.
 * 
 * FTAP: F(lexible) TAP, F(eedback) TAP. A MIDI-based program for serial
 * behavior experiments. FTAP can produce (a class of) auditory sequences,  
 * collect MIDI tapping or musical performance data, and (most 
 * importantly) manipulate feedback  in response to the subjects keystrokes. 
 * The output data is in a text file format which could probably be 
 * converted to a standard midi file by an auxiliary program if necessary.
 * 
 * FTAP itself is primarily designed to run a single (potentially long)
 * experimental trial, based on a parameter file. An interactive mode
 * is also provided for exploration and test, but it should not be
 * used for actual data collection. In particular, parameter changes
 * occurring during a trial will STAY CHANGED.
 * 
 * USAGE: The typical usage will involve a parameter file argument which
 * fully defines the experiment (all paramter values); e.g. "tap <paramfile>".
 * It is possible to follow the parameter file name with some parameter
 * overrides, which will need to be double quoted; these will override
 * the values in the file (e.g., "tap <paramfile> "FEED_DVAL 100" "). It
 * is also possible to invoke the program without any arguments, in which
 * case a set of internal defaults will be used (these will also be used
 * for any parameter which is not explicitly set), and a limited interactive
 * mode is accessible (allows reading in parameter files and setting individual
 * parameter values). Use of this mode is no longer supported. In actual 
 * experimental use, this program will run a
 * single parameter file for a single experiment/trial, and other stuff
 * (randomization, trial numbering, etc) will be done by a wrapper (python
 * or C). Somewhat independent of the use of interactive mode is the
 * STDOUT param, which will not save data for an output file, but will 
 * write key input data (only) immediately to the screen. This avoids
 * the possiblity of overflowing the (currently) 3000 event storage
 * buffer during, e.g., practice sessions.
 *
 * DESIGN: fairly simple in some ways. After setup, the main body of the
 * program is a loop consisting of processing any available input MIDI
 * messages, and then processing any pending output. Since output need not
 * be synchronous with input, output is done via a scheduling queue. When
 * this program is run with (necessary) root privileges, both input and
 * output occur with millisecond granularity (diagnostics are provided in
 * the output file for the user to verify this). In fact, an artifical
 * delay of .5 ms is provided after each check input/check output cycle;
 * otherwise it is impossible to interrupt this program from the keyboard!
 * The program could be speeded up even more if this RTCNap() call were
 * commented out, but if your files don't terminal properly you'll have to
 * hit the power off switch to get your machine back!
 *
 * NOTE: this program, since it runs with real-time priority, is capable
 * of seriously hogging your system. The expectation is that this program
 * will be given (setuid) root privileges only on a dedicated experimental 
 * machine.  It will print out some warnings but will run anyway (WITH
 * POTENTIALLY POOR TIMING RESOLUTION) if it doesn't have proper permissions.
 * 
 * 
 * NOTES: (probably only meaningful to sf).
 * (1) for single sound tapping stuff, to avoid note cutoffs,  may want 
 * to set up keyboard as single note, and actually generate multiple notes
 * in a cycle here (e.g. CDEFG).
 * (2) Time fields may be inconsistent re: longs and ints.
 * (3) Should add (back) delay statistics, to verify random delay?
 * (4) To detect reordering, sequence numbers are assigned to input
 * down (only) strokes, and propagate to associated feedback and event
 * tap_events. This sequencing will probably be totally bogus if multiple
 * feedback channels are being used.
 * (5) Metronome or time based trigger events may possibly occur between
 *  a key down and a key up, potentially leading to screwy behavior
 *  if mapping feedback parameters on a note whose Note_Off feedback
 *  hasn't been triggered yet. Setting velocity to zero by a trigger
 *  is a useful workaround.
 * (6) Output file name format is hardwired to param.sub.trial.xxx.
 * (7) One messy mechanism: the proper reference for timing. UNIX/Linux
 * 	timing is done in milliseconds since the "epoch" (1/1/70), and such
 *	times are necessarily the basis for scheduling. However, specification 
 *	of timing for a trial (output file printing, click files) is most
 *	transparently in terms of milliseconds since trial start. The
 *	transformation between these two forms is based on the trial_starttime
 *	variable. The somewhat arbitrary choice is to do scheduling in
 *	terms of epoch times, and do transformation to trial-start relative
 *	times when necessary; this means that trial_starttime must be
 *	set before start_metron, click files,....etc. (and I've been
 *	burned when this hasn't occurred). Alternatively, the main input
 *	time could trial-start relative, with necessary conversion of 
 *	epoch time occurring at input and output.
 *  (8) This has been coded for UNIX/Linux; sometimes I'll mark such
 * 	dependencies with PORTNOTE in case anyone ever wants to move this
 *	to another OS. This has not been done rigorously, though.
 *  (7) Under no circumstances should this program itself be cluttered
 *	up with "user-friendly" graphical user interface stuff. If such
 * 	an interface is desired, make it a separate program, and then
 *	call FTAP proper.
 *  (8) If graphical stimulus presentation is to be added, it could
 *	probably be localized to  only require calls from the trigger events.
 *	Visual _feedback_ would be messier.
 *  (9) Terminology:  "*EVENT" vs tap_event vs trigger event. Internally, 
 *      most trigger events are now called trig_event rather than event, but
 *	externally (param name) they're still E(VENTS).
 *  (10) Some DEBUG code can be turned on by defines. 
 *  (11) Should perhaps change serious error reports (MIDI_ERROR, MIDISTORE
 *	OVERFLOW, inconsistences in map.c/keypressed flag) to go to standard
 *	error rather than standard out.
 *
 * MIDI DEVICE PROGRAMMING
 * Use of this program requires a midi keyboard and tone generator, and
 * the latter must be programmed for correct behavior. The parameter files
 * allow various specifications of midi channel and midi velocity; the
 * device must be set up to respond properly (e.g., with appropriate
 * volume level) to these signals. Note that velocity sensitivity (loudness 
 * response) on a DX tone generator can depend on carrier output level,
 * EG levels, sensitity, and possibly keyboard velocity scaling. Velocity
 * _attenuates_ the fixed output level. Such MIDI programming is outside
 * the domain of this program.
 * 
 * Internal tone generators might work, but have not been tested.
 *
 * WISH LIST
 *
 *	User-configurable pitch and delay mappings (C code to be linked in).
 *	This could be done by allowing specification of a pitch mapping table
 * 	in the parameter file (with either octave or full keyboard option?)
 *
 *	Specific controls for key up/key down (key release as note on?!).
 *	
 *	Real time (experimenter-controlled) triggers (i.e., triggerd by
 *	experimenter keystroke).
 * 	
 *	Should the user be able to flexibly define output line data? It should
 *	only be data which CHANGES per line (i.e., not subject or trial), and
 *	must be restricted in some way to  system defined-values (time, note,
 *	midichannel, of the tap_event).  Labelling of the fields must be 
 *	explicit.
 *
 *      Make "channels" (particularly split keyboard) more general; it'd be
 *      nice to have separate pitch manipulations on the two sides of the
 *      split, particularly for pitch sequences and pitch lag. Unfortunately,
 *      pitch lag is tricky, because you need to _store_ the data separately
 *      for the two portions of the keyboard!
 * 
 *
 *   LIMITS:
 *	MAXNOTES input events, MAXNOTES feedback/metronome events when
 *	storing to memory (not writing to stdout). Currently 3000.
 *        
 *      MAX_TRIG_EVENTS total trigger events (currently 60).
 *
 *      Some integer events not changeable by trigger events: mask parameters,
 *	stdout, mspb may be funny, changing some feedback channel params
 *	with triggers may result in loss of note off,   ??. See params.c 
 *	for some more details.
 *
 *
 * MODIFICATION HISTORY
 *	3/10/00, sf, MAJOR HACK, generalize to handle both MIDI controllers and 
 * 	keystrokes. Controllers may be necessary for music studies (e.g., wanting
 * 	to let people use the pedal when dealing with), but
 *	they don't really make sense for sophisticated DAF studies. 
 *	The current approach is that controllers are only associated with
 * 	feedback channel 1, and undergo that channel's delay and midi
 * 	channel mapping, but do not undergo pitch or velocity mapping (i.e.,
 *	the controller data itself is not altered). Random delay and
 *	controllers will probably be bogus. Controller input type is 'C';
 * 	output (for lack of any useful one-letter mnenomic) is 'G'
 *	Parameter file changes from earlier versions: remove METRON_CNT
 *	and MIDIPROG; change ENDEXP -> END_EXP, remove unused flag field
 *	from EVENT descriptions and change ID/type ordering.
 *
 *	3/20/00, sf, trigger output now 'T', not 'E', metronome events
 *	take place immediately, store past pitch values in keystroke_pitches 
 *	array so that you can play pitches from N positions back.
 *	Considered putting in read scheduling diagnostics, but given the
 *	design it really shouldn't be different from output scheduling
 *	diagnostics (perpetual read/output cycling).
 *	The immediate metronome trigger code is localized to metron.c,
 *	but requires access to the midistore array, and borrows code from
 *	the keystroke trigger stuff.
 * 
 *	3/28/00, sf, FTAP now sends MIDI Active Sensing every 200
 * 	ms (to fix problem with masking tones and MPU-401); see metron.c
 *
 *	4/3/00, sf, if an internal OVERFLOW is encountered, just stop
 *	saving data, rather than exiting without writing the file. A
 *	little kludgy. Is there an off-by-one error here? Also, make
 *	parameter file open failure a fatal error (except in interactive
 *	mode).
 *
 *	4/10/00, sf, add pitchseq file support (replacing SCALEPITCH and
 *	BACHPITCH).
 *
 *	8/10/00, sf, change the printable MIDI note mapping: I've been
 *	printing out, e.g.,  MIDI note number 84 as C7, but it's apparently
 *	supposed to be C6. NOTE: other changes immediately preceding this
 *	first release, such as use of FEED for FEED1, and the new VMODE 
 *	parameter, have caused me to change the version number to 2.1.
 *
 *	10/17/00, sf: In response to what appear to be (rare!) MIDI hardware 
 *	errors on our system, more extensive MIDI error diagnostics have
 *	been added. Effects here are that the output file name gets printed
 *	to the screen (so users can associate error printouts with an 
 *	output file name), and that the data gets written to a file with
 *	write_log(). For now, it just goes to the data file. See 
 *	linux_midi.c.
 *
 *	12/20/00, sf: Change version number to 2.1.02 for initial web
 *	page release, even though little has changed except for cleaning
 *	up some comments.
 *
 *	12/21/00, sf: modifications for proportional delay. All noted by 
 * 	"#ifdef PPT".
 *
 *	3/5/01, sf: Change version number to 2.1.03 for real "release"
 *	(the BRMIC article is about to be published), although little
 *	has changed except for the doc...
 *
 *	4/19/01, sf, change version number to 2.1.04. Major overhaul of
 *	map.c so that feedback-change triggers between key press and
 *	release get handled correctly.  Fixed both known and newly-
 *	discovered problems. 
 *	NOTE: check_soundnote() will now return 0 for
 *	the NOTE_OFF keystroke of a fixed-length feedback event, so
 *	a small change to sendevent() is also required (as well as
 *	updating of comments).
 *	9/6/01, sf, version number to 2.1.05 (only a small fix to
 *	map.c, but I'd rather keep all changes distinct).
 */


#include <stdio.h>
#include <limits.h>
#include <string.h>
#include <sys/types.h>
#include <sys/times.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <setjmp.h>
#include <errno.h>

	

#include "ftaplimits.h"
#include "tapmidi.h"
#include "params.h"



#ifdef PPT
char version_number[] = "2.1.05.ppt" ;  /* FTAP version # */
#else
char version_number[] = "2.1.05" ;  /* FTAP version # */
#endif



/* time (in ms) of trial start. THIS ENDS UP BEING A CRUCIAL VARIABLE
 * IN A NUMBER OF PLACES, and must be set early on.
 */

unsigned long trial_starttime ;    

int ruid ;			/* uid for file ownership:should be uid_t? */
int rgid ;			/* gid for file ownership */



/********************************************************************
 * file stuff 
 */

/* output file */
#define FILE_SUF ".abs"

FILE *outfile ;

/* filebase is the final component of paramfile (i.e., the paramfile
 * name without the directory), used for output file naming.
 */

char filebase[100] ;	

/* directory of the parameter file (for locating click and pitchseq files) */
char paramdir[150] ;

char tmpfilepath[150] ;



/* parameter file */

char *paramfile = NULL ;
char paramfilebuf[150] ;	/* may be full pathname */




/*************************************************************************
 * storage for keystroke data while running program (don't write to file, to
 * avoid disk I/O while collecting data) 
 */

struct tap_event midistore[MAXNOTES] ;

int midistore_index ;

/* output storage stuff: info saved in sched.c */

extern struct tap_event mdoutstore[] ;
extern int	mdoutstore_index ;


/* separately store the sequence of pitches, for lagpitch mapping */
unsigned char keystroke_pitches[MAXNOTES] ;


/**********************************************************************
 * Stuff related to notes (pitches) played
 */

/* midi note number to string mapping */

#define NNOTES 12
char *notestring[NNOTES] = { "C", "C#", "D", "D#", "E", "F", 
		       "F#","G", "G#", "A", "A#", "B"} ; 

/* For random delay, note _lengths_ should still be as they were played, so
 * need to save delay for each down event. Storage is in this array.
 * Presumably note values will be less than this 
 */

#define MAXKEYS 150

/* storage so that in random delay cases the NoteOff message will be
 * delayed the same amount as the NoteOn message.
 */
int downnote_delay[MAXKEYS] = {0} ;



/********************************************************************
 * General/miscellaneous stuff
 */

char *p ;

#define LINESZ 100
char buf[LINESZ], *bp  ;

/* stats */
extern int delay_cnt, delay_tot ;

/* ONLY if there are no command line args is the program (minimally)
 * interactive.
 */
int  	interactive ;

int	end_experiment ;

/********************************************************************
 *  PORTNOTE: UNIX stuff 
 */

extern int errno ;
void sig_int() ;

#ifdef PPT

/* the delay is calculated from IOI's stored in the following array. Note
 * that even if there's an optimization to store a running total (so
 * you don't calc the mean from scratch each time), you still need the
 * array to know what value is being removed from the calculation!
 */

int	ioi_array[MAXIOI] ;
int	ioi_cnt = 0 ;
unsigned long	lastkeydown = 0 ;	/* this value is checked: must be 0 */
unsigned long	keytime ;		/* should be local, not global */

#endif 


main(argc, argv)
char **argv ;
{	int i, j;

	/* following for interactive mode use only */
	char cmd[50] ;
	char arg[50] ;

	/* date and time to output file */
	time_t	start_time ;	/* PORTNOTE */

	/* in memory input data storage pointer */
	struct tap_event *tsp ;


	/* te_inp is the tap_event structure used for actual note reads, 
	 * which will also be the structure used for the "normal" case of
	 * FCHAN1 output. te_out2p is the tap_event structure used for 
	 * FCHAN2 output. Note that with the ability to split the keyboard,
	 * either of these can legitimately appear without the other. 
 	 * Use and nomenclature is a little confused...
	 */

	struct tap_event *te_inp ;	
	struct tap_event *te_out2p ;	


	int delay ;	/* chosen delay in ms */

	/* sequence number of keystrokes, counting from _1_. Used to
 	 * index into keystroke_pitches array.
 	 */
	int sequenceno ; 

	int note ;



	/* for data merge */
	int tmp_mdos_index  ; 
	struct tap_event *mdosp;


	/* If there are arguments, the first one must be a parameter file.
	 * Subsequent arguments will be specific paramter overrides: a string
 	 * of the form "PARAM VALUE(s)" as a single argument.
	 */
	if (argc > 1) {
		interactive = 0 ;
		strcpy (paramfilebuf, argv[1]) ;
		paramfile = paramfilebuf ;

		if ( set_params(paramfile) == -1) {
			printf("Exiting....\n") ;
			exit (-1) ;
		}
		for (i = 2 ; i < argc ; i++)
			proc_param(argv[i]) ;
	}
	else {	
		interactive = 1 ;
	}
	
	bp = buf ;	/* local/global? Does it matter? */

	/* one-time startup stuff */
	InitRealTime() ;
	init_rand() ;
	ruid = getuid() ;	/* PORTNOTE: get real uid */
	rgid = getgid() ;	/* PORTNOTE: get real gid */



rerun :

	if (interactive) { /* print interactive menu; not supported */
		printf("WARNING: interactive mode should not be" 
			" used for collecting real data\n") ;

menu:		printf("cmd?\tf <PARAMFILE>, p <PARAMSTR>, t <TRIAL>," 
			 "q(uit), r(un), l(ist params):\n");
		fgets (bp, LINESZ, stdin) ; 
 

		/* sloppy, not all may be there */
		sscanf (bp, "%s %s", cmd, arg) ;
		switch (cmd[0]) 
		{ 
	  	case 'f': 	strcpy (paramfilebuf, arg) ;
				paramfile = paramfilebuf ;
				set_params (paramfile) ;

				goto menu ;
	
	
		case 'p': 	/* get the space following the command word */
				p = skip_token(bp) ;
				proc_param(p) ;

				goto menu; 	
	
	  	case 't': 	strcpy (trial_param, arg) ;
				goto menu ;
	
	
		case  'q':	/* exit program */
				exit (0 ) ;

		case 'l':	dump_params(stdout, CURRENT) ;
				goto menu ;

		case 'r':	break ;		/* run a trial! */
				
	
	 	default:	printf("unrecognized command\n") ; 
				goto menu ;; 
		} 
	}
        /* save the initial parameter values so that values in output
	 * file will match the parameter file
	 */
        save_int_params() ;



	/* A bunch of per-trial startup stuff. Normally, an invocation of
	 * FTAP will only run one trial anyway. Could be cleaned up.
 	 *
 	 */

	/*
 	 * Construct output file name (PARAM.SUB.TRIAL) 
 	 */
        paramdir[0] = 0 ;
        if (paramfile)   {
                int tmplen ;
                /* last component only for filename: PORTNOTE,
                 * pathname dependency
                 */

                if ((p = strrchr (paramfile, '/')) != NULL) {
                        strcpy (filebase, p + 1) ;
                        /* include the trailing '/' */
                        tmplen = p - paramfile + 1 ;

                        strncpy (paramdir, paramfile, tmplen) ;
                        paramdir[tmplen] = 0 ;
                }
                else
                        strcpy (filebase,  paramfile) ;
        }
        else
                strcpy (filebase, "nofile") ;
        strcat (filebase, ".") ;
        strcat (filebase, sub_param ) ;
        strcat (filebase, ".") ;
        strcat (filebase, block_param ) ;
        strcat (filebase, ".") ;
        strcat (filebase, trial_param) ;



	/* Set up for handling output (to screen/file). Stdout cabability
	 * may add more complexity than it's
	 */
	if (stdout_param)
		outfile = stdout ;
	else {
		strcpy (bp, filebase) ;
		strcat(bp, FILE_SUF) ;
		outfile = fopen (bp, "w") ;
		if (outfile != NULL) {
			/* give file ruid and rgid in case running as
			 * setuid root.
			 */
			chown (bp, ruid, rgid) ;	
		}
	}
	if (outfile == NULL) {
		printf("Error: cannot open output file %s\n", bp) ;
		exit (-1) ;
	}
	if (stdout_param)
		printf("Output to stdout\n") ;
	else
		printf("Output to %s\n", bp) ;
	
	/* get info for time and data for saving in output file.
	 */  

	start_time = time(NULL) ;


	/* a bunch of per trial initialization */

	init_te_queue() ;	/* reset all scheduling queues */
	reset_pitch_indexes() ;
	/* initialize data structures for metronome and keystroke
	 * events
	 */
	ev_init() ;
	InitSaveBuf(); 		/* for making sure all notes get turned off: 
				 * linux_midi.c */
	InitMidi() ;
	midistore_index = 0 ;
	sequenceno = 0 ;
	end_experiment = 0 ;

	/* stats for delay: are these also still in output.c? */
	delay_tot = 0 ;
	delay_cnt = 0 ;

	sync() ;	/* do any pending OS disk I/O  */

	/* read in the pitch sequence file, if any */
        if (strlen (pitchseq_file_param) > 0) {
                strcpy (tmpfilepath, paramdir) ;
                strcat (tmpfilepath, pitchseq_file_param) ;
                ReadPitchSeqFile(tmpfilepath) ;
        }


	/* NOTE:  Lots of things depend on  start time of the trial: metronome 
         * scheduling, click file scheduling, ..., so SETTING OF 
	 * trial_starttime MUST OCCUR EARLY..
 	 *
	 * I forget why this isn't used for start_time above...
	 */

	trial_starttime = gettodms() ;

	
        /* setup prescheduled click file events, if any... the 
         * click files must be in the same directory as the parameter
         * file (or at least relative to that directory).
 	 *
         * Must  be called  after trial_starttime is set.
         */
        if (strlen (click1_file_param) > 0) {
                strcpy (tmpfilepath, paramdir) ;
                strcat (tmpfilepath, click1_file_param) ;
                sched_click (tmpfilepath, click1_offset_param) ;
        }
        if (strlen (click2_file_param) > 0) {
                strcpy (tmpfilepath, paramdir) ;
                strcat (tmpfilepath, click2_file_param) ;
                sched_click (tmpfilepath, click2_offset_param) ;
        }



	printf("hit INTR (ctl-c) to stop....\n") ;



	/* PORTNOTE: UNIX */
	signal(SIGINT, sig_int) ;



	/* schedule onset of masking noise (if any) and any timing events.
	 */
	mask_on() ;
	ev_sched_time() ;


	/* set up for scheduling of the pacing notes. NOTE: this _must_ 
	 * occur after trial_starttime is set. 
	 */
	start_metronome () ;	

	InitOutput();
	StartRTC() ;

	/* The main loop: look for input midi events, process them if any,
 	 * then call output scheduler. Lots of other stuff goes on.  Some
	 * of this got messy when I added flag-type encoding to the
	 * event structure.
 
	 * Exiting from the experiment is based on setting of the flag
	 * end_experiment, which will happen either with a user ctrl-C
 	 * keyboard interrupt (possibly detected in read_midi_event() ), or 
         * via a trigger event set to END_EXP.
	 */

	while (!end_experiment) {
			
		while ( (te_inp = ReadMidiEvent()) != NULL) {  

			/* input NOTE events are necessarily keystrokes */
			if ((te_inp->te_type & TE_KEYSTROKE) && 
			    te_inp->inevent.midinote_onoff == NOTEDOWN) {
				sequenceno++ ;
				/* save the pitch for pitchlag mapping */ 
				keystroke_pitches[sequenceno] = 
					te_inp->inevent.midinote_pitch ;
				te_inp->te_sequenceno = sequenceno ;

#ifdef PPT
				/* place the IOI in the circular buffer. Logical
				 * buffer size is set by the delayppt_n
				 * parameter, which better not exceed the
				 * static maximum! IOI's outside the specified
				 * range don't count for the delay.
				 */
				keytime = te_inp->inevent.time ;
				if (lastkeydown != 0) {   /*not first keystroke*/
					int diff, index ;
					diff = keytime - lastkeydown ;
					if (diff <= delayppt_uplim_param &&
						diff >= delayppt_lowlim_param) {
						index = ioi_cnt%delayppt_n_param;
						ioi_array[index] = diff ;
						ioi_cnt++ ;
#ifdef PPTDEBUG
						{
						int j, k ;
						printf("\nPPTDEBUG: ftap.c, ioi_cnt %d, ioi_array = \n", 
							ioi_cnt) ;
						for (j = 0 ; j < MAXIOI; j++)
							printf("%d  ", ioi_array[j]) ;
		
						}
#endif (PPTDEBUG)
					}
				
				}
				lastkeydown = keytime ; /* .... always ...*/
#endif (PPT)
				

				/* key events take place before any mappings */
				while ((i = is_key_event(sequenceno)) >= 0) {
					ev_exec_action (i) ;
	
					/* It is desirable to have the
					 * trigger  event written to the output
					 * file, but there's no tap_event for 
					 * it.	Just fill in an mdstore entry
					 * (will not go to stdout), making 
					 * special use of available fields.
					 */

					tsp = &(midistore[midistore_index]) ;
					if (++midistore_index > MAXNOTES){
						printf("ERROR: MIDISTORE OVERFLOW!!\n");
						midistore_index-- ;
						continue ;
					}
					tsp->te_type |= TE_TRIGGER;
					/* Keystroke triggers as input, others
					 * as output (????)
					 */
					tsp->te_io = TE_INPUT ;
					tsp->te_sequenceno = sequenceno ;
					/* for good measure, write the index */
					tsp->inevent.midinote_vel = i ;
					tsp->inevent.midinote_pitch = 
						trig_event_table[i].id;
					tsp->inevent.time = te_inp->inevent.time ;
					tsp->inevent.midinote_onoff = TRIG_KEY_TYPE ;
				}
			}
			else 
				te_inp->te_sequenceno = 0 ;	/* no sequence for up */
			/* Non-zero sequence numbers only get allocated
			 * to keystroke down events (and these, by default,
			 * are duplicated on the keystroke trigger event).
			 */
	
	
#if DEBUG2
		printf("Got a note: ") ;
		PrintTapEvent(te_inp) ;
#endif
	
			/* record the input keystroke data */
			if (stdout_param) {
				write_data(te_inp) ;
			}
			else { 	/* save to memory.
				 */
				tsp = &(midistore[midistore_index]) ;
				if (++midistore_index > MAXNOTES){
					printf("ERROR: MIDISTORE OVERFLOW!!\n");
					midistore_index-- ;
				}
				else
					copy_te (te_inp, tsp) ;
			}
	
	
			
	/* Feedback channels: Under some
	 * circumstances (IAF + DAF, or different alterations for each hand),
	 * there is  possible use for for 2 feedback "channels". 
	 * te_inp, read in above, will also be used as the output event 
	 * for FCHAN 1. If FCHAN 2 is also used, another tap_event will be 
	 * allocated (te_out2p), which will be initialized from te_inp.
	 * Both or either of these channels may end up sounding an event.
	 * Slight bogosity: some of this mapping work is unneeded for the
	 * up event of a fixed-length feedback response.
	 */
	
#if DEBUG3
printf("INPUT NOTE (te_inp)\n") ; 	
PrintTapEvent(te_inp) ;
#endif
			/* do FCHAN2 first, since FCHAN1 may free te_inp if 
			 * it's an upnote event for a fixed-length feedback 
			 * pulse (which has already been sent). Note that 
			 * this NOTEOFF case is now checked by check_soundnote,
			 * rather than in the sendevent routine.
			 * rather, it is handled in the sendevent routine.
 			 * By definition, no MIDI controllers for FCHAN2.
	 	 	 * FIXME NOTE: check that tap_event structures are 
			 * getting allocated and freed correctly.
 			 * 
 			 * Input NOTE's are necessarily KEYSTROKES
	 		 */
			if (te_inp->te_type & TE_NOTE && 
				check_soundnote (FCHAN2, te_inp))  {
#if DEBUG3
printf("SOUNDING EVENT ON FCHAN2\n") ;
#endif
				te_out2p = get_te() ;
				copy_te (te_inp, te_out2p) ;
				/* FIXME: flags */
				te_out2p->te_io = TE_OUTPUT;	
				mapevent (FCHAN2, te_out2p) ;
				sendevent (FCHAN2, te_out2p) ;
			} ;
			if (check_soundnote (FCHAN1, te_inp)) {
#if DEBUG3
printf("SOUNDING EVENT ON FCHAN1\n") ;
#endif
				te_inp->te_io = TE_OUTPUT;	
				mapevent (FCHAN1, te_inp) ;
				sendevent (FCHAN1, te_inp) ;
			}
			else { /* no auditory feedback for this channel, 
				* just free  tap_event 
				*/
				free_te (te_inp) ;
			};	 
		}	/* end of ReadMidiEvent loop */

		ProcOutput() ;	/* do output scheduling */

		RTCNap() ;	/* allow something else to occur for .5 ms */
	
	}	/* end of experiment) */
	printf ("\nExperiment ended\n") ;
	mask_off() ;
	EndOutput() ;
	StopRTC() ;
	CloseMidi() ;


	/* write saved data to file, along with delayed header 
 	 * MERGE the two sources of data info, so it comes out in order (this
	 * could be done in post-processing, but...). If mdoutstore info has
	 * not been saved, the corresponding index should be 0.
	 * Algorithm: go through each item of midistore, writing out any
	 * out events which precede it in time, and then writing the 
	 * midistore item. At end, write out any leftover out events.
	 * This is sort of ugly/messy.
	 */

	/* If there is to be user-controlled filtering of printing 
	 * metronome/feedback data, should it be done here, or in sched? 
	 */

	/* midistore[] = keystroke record = input events, + keystroke triggers.
	 * mdoutstore = metronome/feedback = output events + metronome/time
	 * triggers.
	 */
	if (!stdout_param) {
		fprintf (outfile, "# TIME\t\t%s", ctime(&start_time)) ;
		/* must do this code _before_ dump_params... */
		if (delay_cnt == 0)
			sprintf (av_delay_param, "%.2f", 0.0) ;
		else		
			sprintf (av_delay_param, "%.2f", 
				(float)delay_tot/delay_cnt) ;

		/* write _all_ the parameter values */
		dump_params (outfile, INITIAL) ;
		write_log(outfile) ;
		write_header() ;
		tmp_mdos_index = 0 ;

		for (i = 0 ; i < midistore_index ; i++) {
			tsp = &(midistore[i]) ;
			while (tmp_mdos_index < mdoutstore_index) {
				mdosp = &(mdoutstore[tmp_mdos_index]) ;
				if (mdosp->outevent.time < tsp->inevent.time) {
					write_data(mdosp) ;
					tmp_mdos_index++ ;
				}
				else
					break ;	/* from while loop */
			}
			write_data(tsp) ;
		}

		/* finish up any leftover outdata */
		while (tmp_mdos_index < mdoutstore_index) {
			write_data(&(mdoutstore[tmp_mdos_index]));
			tmp_mdos_index++ ;
		}
		fclose (outfile) ;
	}
	
	if (interactive)
		goto rerun ;	/* user-controlled exit happens at menu */
	else
		exit(0) ;
}


/* signal catcher for keyboard termination */
/* PORTNOTE: UNIX */

void sig_int()
{	end_experiment = 1 ;
}

/***********************************************************************
 * Miscellaneous functions
 */

/**************************
 * Code which is shared for both feedback channels. Controller events
 * only undergo delay and channel mapping.  Note that type has already
 * been set to the _output_ mode. Triggers shouldn't show up here.
 */

mapevent (channel, tep)
int channel ;
struct tap_event *tep ;
{
#if DEBUG3
printf("mapevent(): channel %d\n", channel+1) ;
printf("mapevent(): tap event on input:\n") ;
PrintTapEvent (tep) ;

#endif
	if (tep->te_type & TE_NOTE) {
		tep->outevent.midinote_onoff =  
			tep->inevent.midinote_onoff ; /* not mapped */
		pitchmap (channel, tep) ;
		velmap (channel, tep) ;
	}
	else	{	/* Controller */
		tep->outevent.status =
			tep->inevent.status ; 
		tep->outevent.byte1 =
			tep->inevent.byte1 ; 
		tep->outevent.byte2 =
			tep->inevent.byte2 ; 
	}
		
	delaymap (channel, tep) ;
	chanmap (channel, tep) ;

	/* te_type is set outside, as tentative preparation for metronome
	 * mapping 
	 */
#if DEBUG3
printf("mapevent(): tap event on output:\n") ;
PrintTapEvent (tep) ;
#endif
}
	
/* write the event to the output queue, taking care of allocating the
 * key up event if we're doing fixed-length mapping.
 * The scheduler will eventually free any tap_events placed
 * on the scheduling queue.
 * NOTE: if subtype is guaranteed to be initialized to zero, the NOTE check
 * could probably be simplified to subtype KEYSTROKE check...
 */

sendevent (channel, tep)
int channel ;
struct tap_event *tep ;
{
	/* structure for the up event of a fixed pulse */
	struct tap_event *te_out2p ;
	int len_param ;

#if DEBUG3
printf("sendevent: channel %d\n", channel+1) ;
#endif
	len_param = * (fchan_params[channel] [FCHAN_LEN_INDEX]) ;

	/* If a fixed length note, will need to generate the note off */
	if (tep->te_type & TE_KEYSTROKE && len_param && 
		tep->outevent.midinote_onoff == NOTEDOWN) {
#if DEBUG3
printf("FCHAN1: Putting fixed length event in queue\n") ; 
PrintTapEvent(tep) ;
#endif
			te_out2p = get_te() ;
			copy_te (tep, te_out2p) ;
			te_out2p->outevent.time = tep->outevent.time +
				len_param ;
			te_out2p->outevent.midinote_onoff = NOTEUP ;
			te_out2p->outevent.midinote_vel = 0 ;
			te_out2p->te_sequenceno = 0 ;
                       	insert_event (tep) ;
                       	insert_event (te_out2p) ;
	}
	else {	/* follow user's input on/off */
#if DEBUG3
printf("FCHAN1: Putting following length event in queue\n") ; 	/* XXX */
PrintTapEvent(tep) ;
#endif
       		insert_event (tep) ;
	}
		
}


/**************************
 * Routines using global outfile to do some formatting for the output
 * file (be it stdout or on disk).
 */

write_header()
{
	fprintf(outfile,"# Stamp\tUpDown\tCH\tNote#\tNoteNm\tVel\tSeq\tType\n"); 
}


/* 
 * This routine works, but is structurally a mess. _I_ can barely understand it.
 * It gets complicated because there are 3 fundamentally different types of
 * events to be printed (Notes, Triggers, and Controllers), all encoded 
 * in the same data structure. Also, the new factorization of the 'type'
 * fields in struct tapevent simplifies life elsewhere at the cost
 * of complication here.
 *
 * The routine to print out an output line, given a tap event. The main
 * case (keystroke, feedback, and metronome events) is what this was
 * designed for (it's what the column labels provide), and these are
 * straightforward. However, it also seems necessary to provide full
 * information for MIDI controllers (damper pedal at the least, and
 * the others sort of fall out), and it is useful for diagnostic
 * purposes to include trigger events in the output file. These last
 * two end up being messy. 
 *
 * The output format (except for the last two fields) is differentiated by
 * te_type = NOTE, CONTROLLER, or TRIGGER. KEYSTROKE and METRONOME events
 * are the main types of NOTE.
 *
 * Output line format:
	1st field: Integer: always time
	2nd field: Character: U/D for NOTE:, trigger type (T, M, K) for 
		TRIGGER, X for CONTROLLER
	3rd field: Integer:  Midi Channel for NOTE and CONTROLLER, 0 
		for TRIGGER.
	4th field: Integer:  MIDI pitch value for NOTE, 1st data byte 
		for CONTROLLER, 
		internal index for TRIGGER (useless)
	5th field: String:  Ascii representation  of pitch for NOTE, 
 		MIDI status nibble for controllers (so, e.g., Pitch Bend 
		can be differentiated from pedal if necessary), "--" for 
		triggers.
	6th field: Integer: Velocity for NOTE, 2nd data byte for CONTROLLER,
		trigger EVENT ID for triggers.
	7th field: Integer:  Keystroke Sequence number for  KEYSTROKE down 
		events, as well as for the  associated feedback (keystroke 
		output); 0 in all other cases. 
	8th field: A single (usually mnemonic)  character identifying 
		the event type. 
		'K': Keystroke (input) [NOTE]
		'F': Feedback for a keystroke (output) [NOTE]
		'M': Metronome (output) [NOTE]
		'J': Junk (e.g., masking noise) (output) [NOTE]
		'C': MIDI Controller (input) [CONTROLLER]
		'G': MIDI Controller (output) [CONTROLLER]
		'T': Internal trigger event (??) [TRIGGER]
		
  Is INPUT/OUTPUT redundant now? 

 */

write_data(tep)
struct tap_event *tep ;
{
	struct midi_event *mep ;

	/* the fields which need to be special  */
	int  note, vel ;
	char updown, event_type ;
	char notestringbuf[10];

	int type ;
	char io ;


	type = tep->te_type ;
	io = tep->te_io ;
#if DEBUG3
	printf ("write data, type = %x%x, tep = \n", type) ;
	PrintTapEvent (tep) ;
#endif

	if (io == TE_INPUT)
		mep = & (tep->inevent) ;
	else if (io == TE_OUTPUT)
		mep = & (tep->outevent) ;

	if (type & TE_CONTROLLER)
		updown = 'X';
	else
		if (mep->midinote_onoff == NOTEDOWN)
			updown =  NOTEDOWN_CHAR ;
		else if (mep->midinote_onoff == NOTEUP)
			updown =  NOTEUP_CHAR ;
		else { 	/* better be trigger event! */
			updown = mep->midinote_onoff ;
		}


	if (type & TE_NOTE) 
		sprintf(notestringbuf, "%s%d", 
			notestring[mep->midinote_pitch % NNOTES],
			mep->midinote_pitch/NNOTES - 1) ;
	
	else if  (type & TE_CONTROLLER)
		sprintf(notestringbuf, "%X", mep->status & 0xF0) ;
	else if (type & TE_TRIGGER)
		strcpy (notestringbuf, "--") ;	/* NADA */
	else {	
		printf("write_data: unknown type! 0x%x\n", type) ;
		exit (-1) ;
	}


	/* not distinguishing controllers, notes, or triggers here */
	note = mep->byte1 ;
	vel = mep->byte2 ;

	/* this used to be so simple, sigh ... */
	if (type & TE_NOTE) 
		switch (type & TE_SUBTYPE_MASK)
		{
		 	case TE_KEYSTROKE: if (io ==  TE_INPUT)
						event_type = 'K' ;
					    else
						event_type = 'F' ;
				            break ;
			case TE_METRON :   event_type = 'M' ;
				    break ;

			case TE_JUNK :     event_type = 'J' ;		
				    break ;
	
		}
	if (type & TE_CONTROLLER)
		if (io == TE_INPUT)
			event_type = 'C' ;
		else
			event_type = 'G' ;
	if (type & TE_TRIGGER)
		event_type = 'T' ;
	
		
      	fprintf(outfile, "%lu\t%c\t%d\t%d\t%s\t%d\t%d\t%c\n",
      		mep->time - trial_starttime,  updown,
		mep->midichan, note, notestringbuf,
		vel, tep->te_sequenceno, event_type) ; 

}

